<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Entities\TenBuilding;
use Modules\Mosque\Entities\TenFloor;
use Modules\Mosque\Entities\TenRoom;
use Modules\Mosque\Entities\TenSeat;
use Modules\Mosque\Utils\MosqueAuditUtil;

class TenancySeatBoardController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if ($businessId <= 0) {
            abort(403, 'Unauthorized action.');
        }
        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (auth()->user()->can('superadmin')) {
            return;
        }
        if (! auth()->user()->hasAnyPermission(['mosque.tenancy.view', 'mosque.tenancy.allocate', 'mosque.tenancy.edit'])) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function ensureTenancyEnabled(int $businessId): void
    {
        if (! Schema::hasTable('mosque_settings')) {
            return;
        }
        $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        if (! (bool) ($settings['tenancy_enabled'] ?? true)) {
            abort(403, 'Tenancy is disabled.');
        }
    }

    public function index()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $buildings = Schema::hasTable('ten_buildings')
            ? TenBuilding::query()->where('business_id', $businessId)->orderBy('name')->pluck('name', 'id')->toArray()
            : [];

        return view('mosque::tenancy.seat_board', compact('buildings'));
    }

    public function floors(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $buildingId = (int) $request->input('building_id', 0);
        if ($buildingId <= 0) {
            return ['success' => true, 'floors' => []];
        }

        $floors = TenFloor::query()
            ->where('business_id', $businessId)
            ->where('building_id', $buildingId)
            ->orderBy('sort_order')
            ->orderBy('name_or_number')
            ->get(['id', 'name_or_number']);

        return ['success' => true, 'floors' => $floors];
    }

    public function floorData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $floorId = (int) $request->input('floor_id', 0);
        if ($floorId <= 0) {
            return ['success' => true, 'rooms' => []];
        }

        // Fetch rooms and seats in one go. Avoid eager-loading heavy JSON; only required fields.
        $rooms = TenRoom::query()
            ->where('business_id', $businessId)
            ->where('floor_id', $floorId)
            ->orderBy('code')
            ->get(['id', 'code', 'type', 'capacity_total', 'notes']);

        $roomIds = $rooms->pluck('id')->all();
        if (empty($roomIds)) {
            return ['success' => true, 'rooms' => []];
        }

        $today = now()->toDateString();
        $soon = now()->addMonth()->toDateString();

        $seats = DB::table('ten_seats as s')
            ->leftJoin('ten_allocations as a', function ($join) use ($businessId) {
                $join->on('a.seat_id', '=', 's.id')
                    ->where('a.business_id', '=', $businessId)
                    ->whereNull('a.deleted_at')
                    ->where('a.status', '=', 'active');
            })
            ->leftJoin('ten_tenants as t', function ($join) use ($businessId) {
                $join->on('t.id', '=', 'a.tenant_id')
                    ->where('t.business_id', '=', $businessId)
                    ->whereNull('t.deleted_at');
            })
            ->where('s.business_id', $businessId)
            ->whereNull('s.deleted_at')
            ->whereIn('s.room_id', $roomIds)
            ->orderBy('s.room_id')
            ->orderBy('s.code')
            ->select([
                's.id',
                's.room_id',
                's.code',
                's.capacity',
                's.status',
                's.leaving_on',
                's.notes',
                'a.id as allocation_id',
                'a.start_date as alloc_start_date',
                'a.end_date as alloc_end_date',
                'a.rent_amount as alloc_rent_amount',
                'a.billing_cycle as alloc_billing_cycle',
                't.id as tenant_id',
                't.full_name as tenant_name',
                't.phone as tenant_phone',
                't.photo_path as tenant_photo_path',
                't.form_data as tenant_form_data',
            ])
            ->get();

        $groupedSeats = [];
        foreach ($seats as $s) {
            $leavingOn = $s->leaving_on ? (string) $s->leaving_on : null;
            $allocEnd = $s->alloc_end_date ? (string) $s->alloc_end_date : null;
            $isLeavingSoon = ($leavingOn !== null && $leavingOn >= $today && $leavingOn <= $soon)
                || ($allocEnd !== null && $allocEnd >= $today && $allocEnd <= $soon);

            $rawStatus = (string) ($s->status ?? 'available');
            if ($rawStatus === 'maintenance') {
                $effective = 'maintenance';
            } elseif ($rawStatus === 'held') {
                $effective = 'held';
            } elseif ($isLeavingSoon) {
                $effective = 'leaving';
            } elseif (! empty($s->allocation_id)) {
                $effective = 'occupied';
            } else {
                $effective = 'available';
            }

            $area = '';
            try {
                $fd = is_string($s->tenant_form_data ?? null) ? json_decode((string) $s->tenant_form_data, true) : (is_array($s->tenant_form_data ?? null) ? $s->tenant_form_data : []);
                $area = (string) (($fd['district'] ?? '') ?: ($fd['local_area'] ?? '') ?: '');
            } catch (\Throwable $e) {
            }

            $monthsStayed = 0;
            try {
                if (! empty($s->alloc_start_date)) {
                    $monthsStayed = now()->diffInMonths(\Carbon\Carbon::parse((string) $s->alloc_start_date));
                }
            } catch (\Throwable $e) {
            }

            $photoUrl = '';
            try {
                if (! empty($s->tenant_photo_path) && ! empty($s->tenant_id)) {
                    $photoUrl = route('mosque.tenancy.tenants.file', ['id' => (int) $s->tenant_id, 'path' => (string) $s->tenant_photo_path], false);
                }
            } catch (\Throwable $e) {
            }

            $groupedSeats[(int) $s->room_id][] = [
                'id' => (int) $s->id,
                'code' => (string) $s->code,
                'capacity' => (int) ($s->capacity ?? 1),
                'status' => (string) ($s->status ?? 'available'),
                'effective_status' => $effective,
                'leaving_on' => $leavingOn,
                'notes' => (string) ($s->notes ?? ''),
                'allocation' => ! empty($s->allocation_id) ? [
                    'id' => (int) $s->allocation_id,
                    'start_date' => (string) ($s->alloc_start_date ?? ''),
                    'end_date' => (string) ($s->alloc_end_date ?? ''),
                    'rent_amount' => (float) ($s->alloc_rent_amount ?? 0),
                    'billing_cycle' => (string) ($s->alloc_billing_cycle ?? 'monthly'),
                    'months_stayed' => $monthsStayed,
                ] : null,
                'tenant' => ! empty($s->tenant_id) ? [
                    'id' => (int) $s->tenant_id,
                    'name' => (string) ($s->tenant_name ?? ''),
                    'phone' => (string) ($s->tenant_phone ?? ''),
                    'photo' => (string) ($s->tenant_photo_path ?? ''),
                    'photo_url' => $photoUrl,
                    'area' => $area,
                ] : null,
            ];
        }

        $payloadRooms = [];
        foreach ($rooms as $r) {
            $payloadRooms[] = [
                'id' => (int) $r->id,
                'code' => (string) $r->code,
                'type' => (string) $r->type,
                'capacity_total' => $r->capacity_total !== null ? (int) $r->capacity_total : null,
                'notes' => (string) ($r->notes ?? ''),
                'seats' => $groupedSeats[(int) $r->id] ?? [],
            ];
        }

        return ['success' => true, 'rooms' => $payloadRooms];
    }

    public function seatStatus(Request $request, $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        if (! auth()->user()->can('mosque.tenancy.edit')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'status' => 'required|in:available,held,maintenance',
        ]);

        $seat = TenSeat::query()
            ->where('business_id', $businessId)
            ->findOrFail((int) $id);

        $hasActiveAlloc = Schema::hasTable('ten_allocations') && $seat->activeAllocation()->exists();
        $status = (string) $request->input('status');

        if ($hasActiveAlloc && $status !== 'maintenance') {
            return ['success' => false, 'msg' => __('mosque::mosque.tenancy_err_seat_has_allocation')];
        }

        $old = $seat->toArray();
        $seat->status = $status;
        if ($status !== 'available') {
            // leave leaving_on as-is
        } else {
            $seat->leaving_on = null;
        }
        $seat->save();

        MosqueAuditUtil::log($businessId, 'update', 'ten_seat', (int) $seat->id, [
            'old' => $old,
            'new' => $seat->toArray(),
        ]);

        return ['success' => true, 'msg' => __('lang_v1.success')];
    }
}
