<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Utils\MosqueAuditUtil;

class TenancySettingsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if ($businessId <= 0) {
            abort(403, 'Unauthorized action.');
        }
        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (auth()->user()->can('superadmin')) {
            return;
        }
        if (! auth()->user()->can('mosque.tenancy.settings.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $row = Schema::hasTable('mosque_settings')
            ? MosqueSetting::query()->firstOrCreate(['business_id' => $businessId], ['settings' => []])
            : null;

        $settings = $row?->settings ?: [];

        return view('mosque::tenancy.settings', compact('settings', 'row'));
    }

    public function update(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $request->validate([
            'tenancy_enabled' => 'nullable|boolean',
            'tenancy_billing_mode' => 'required|in:auto_lazy,manual',
            'tenancy_notice_sms_enabled' => 'nullable|boolean',
            'tenancy_notice_whatsapp_enabled' => 'nullable|boolean',
            'tenancy_notice_email_enabled' => 'nullable|boolean',
            'tenancy_template_bill_generated_sms' => 'nullable|string|max:1000',
            'tenancy_template_bill_generated_whatsapp' => 'nullable|string|max:2000',
            'tenancy_template_bill_generated_email_subject' => 'nullable|string|max:255',
            'tenancy_template_bill_generated_email' => 'nullable|string|max:4000',
            'tenancy_template_allocation_approved_whatsapp' => 'nullable|string|max:2000',
            'tenancy_qr_title' => 'nullable|string|max:120',
            'tenancy_qr_primary_color' => 'nullable|string|max:20',
            'tenancy_qr_ttl_minutes' => 'nullable|integer|min:1|max:525600',
        ]);

        if (! Schema::hasTable('mosque_settings')) {
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }

        $row = MosqueSetting::query()->firstOrCreate(['business_id' => $businessId], ['settings' => []]);
        $settings = $row->settings ?: [];
        $old = $settings;

        $settings['tenancy_enabled'] = (bool) $request->boolean('tenancy_enabled');
        $settings['tenancy_billing_mode'] = (string) $request->input('tenancy_billing_mode', 'manual');
        $settings['tenancy_notice_sms_enabled'] = (bool) $request->boolean('tenancy_notice_sms_enabled', true);
        $settings['tenancy_notice_whatsapp_enabled'] = (bool) $request->boolean('tenancy_notice_whatsapp_enabled', true);
        $settings['tenancy_notice_email_enabled'] = (bool) $request->boolean('tenancy_notice_email_enabled', false);

        $settings['tenancy_template_bill_generated_sms'] = (string) $request->input('tenancy_template_bill_generated_sms', '');
        $settings['tenancy_template_bill_generated_whatsapp'] = (string) $request->input('tenancy_template_bill_generated_whatsapp', '');
        $settings['tenancy_template_bill_generated_email_subject'] = (string) $request->input('tenancy_template_bill_generated_email_subject', '');
        $settings['tenancy_template_bill_generated_email'] = (string) $request->input('tenancy_template_bill_generated_email', '');
        $settings['tenancy_template_allocation_approved_whatsapp'] = (string) $request->input('tenancy_template_allocation_approved_whatsapp', '');

        $settings['tenancy_qr_title'] = (string) $request->input('tenancy_qr_title', '');
        $settings['tenancy_qr_primary_color'] = (string) $request->input('tenancy_qr_primary_color', '#000000');
        $settings['tenancy_qr_ttl_minutes'] = (int) $request->input('tenancy_qr_ttl_minutes', 60 * 24 * 30);

        $row->settings = $settings;
        $row->save();

        MosqueAuditUtil::log($businessId, 'update', 'tenancy_settings', null, [
            'old' => $old,
            'new' => $settings,
        ]);

        return ['success' => true, 'msg' => __('lang_v1.success')];
    }
}
