<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Entities\TenBuilding;
use Modules\Mosque\Entities\TenFloor;
use Modules\Mosque\Entities\TenRoom;
use Modules\Mosque\Entities\TenSeat;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Yajra\DataTables\Facades\DataTables;

class TenancyStructureController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if ($businessId <= 0) {
            abort(403, 'Unauthorized action.');
        }
        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (auth()->user()->can('superadmin')) {
            return;
        }
        if (! auth()->user()->can('mosque.tenancy.settings.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function ensureTenancyEnabled(int $businessId): void
    {
        if (! Schema::hasTable('mosque_settings')) {
            return;
        }
        $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        if (! (bool) ($settings['tenancy_enabled'] ?? true)) {
            abort(403, 'Tenancy is disabled.');
        }
    }

    public function buildingsList(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        if (! Schema::hasTable('ten_buildings')) {
            return ['success' => true, 'buildings' => []];
        }

        $buildings = TenBuilding::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->get(['id', 'name']);

        return ['success' => true, 'buildings' => $buildings];
    }

    public function floorsList(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $buildingId = (int) $request->input('building_id', 0);
        if ($buildingId <= 0 || ! Schema::hasTable('ten_floors')) {
            return ['success' => true, 'floors' => []];
        }

        $floors = TenFloor::query()
            ->where('business_id', $businessId)
            ->where('building_id', $buildingId)
            ->orderBy('sort_order')
            ->orderBy('name_or_number')
            ->get(['id', 'name_or_number']);

        return ['success' => true, 'floors' => $floors];
    }

    public function roomsList(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $floorId = (int) $request->input('floor_id', 0);
        if ($floorId <= 0 || ! Schema::hasTable('ten_rooms')) {
            return ['success' => true, 'rooms' => []];
        }

        $rooms = TenRoom::query()
            ->where('business_id', $businessId)
            ->where('floor_id', $floorId)
            ->orderBy('code')
            ->get(['id', 'code']);

        return ['success' => true, 'rooms' => $rooms];
    }

    public function buildingsShow($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $building = TenBuilding::query()->where('business_id', $businessId)->findOrFail((int) $id);
        return ['success' => true, 'data' => $building->only(['id', 'name', 'address', 'notes'])];
    }

    public function floorsShow($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $floor = TenFloor::query()->where('business_id', $businessId)->findOrFail((int) $id);
        return [
            'success' => true,
            'data' => $floor->only(['id', 'building_id', 'name_or_number', 'sort_order']),
        ];
    }

    public function roomsShow($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $room = TenRoom::query()->where('business_id', $businessId)->findOrFail((int) $id);
        return [
            'success' => true,
            'data' => $room->only(['id', 'floor_id', 'code', 'type', 'notes']),
        ];
    }

    public function seatsShow($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $seat = TenSeat::query()->where('business_id', $businessId)->findOrFail((int) $id);
        return [
            'success' => true,
            'data' => $seat->only(['id', 'room_id', 'code', 'status']),
        ];
    }

    public function buildingsData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $q = TenBuilding::query()->where('business_id', $businessId)->select(['id', 'name', 'address', 'notes']);
        return DataTables::of($q)
            ->addColumn('action', function ($row) {
                $edit = '<button class="btn btn-xs btn-primary ten-struct-edit" data-type="building" data-id="'.(int) $row->id.'"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $deleteUrl = route('mosque.tenancy.structure.buildings.destroy', [(int) $row->id], false);
                $delete = '<button data-href="'.$deleteUrl.'" onclick="(function(){var e=(arguments&&arguments[0])?arguments[0]:(window.event||null);if(e){if(e.preventDefault)e.preventDefault();if(e.stopImmediatePropagation)e.stopImmediatePropagation();else if(e.stopPropagation)e.stopPropagation();}var url=\''.$deleteUrl.'\';var ent=\'building\';if(typeof window.mosqueOpenDeleteConfirm===\'function\'){window.mosqueOpenDeleteConfirm(url,\'DELETE\',ent);return;}window.location.reload();})(); return false;" class="btn btn-xs btn-danger delete_mosque_building"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function buildingsStore(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'name' => 'required|string|max:255',
            'address' => 'nullable|string|max:1000',
            'notes' => 'nullable|string|max:2000',
        ]);

        $building = TenBuilding::query()->create([
            'business_id' => $businessId,
            'name' => (string) $request->input('name'),
            'address' => (string) $request->input('address', ''),
            'notes' => (string) $request->input('notes', ''),
        ]);

        MosqueAuditUtil::log($businessId, 'create', 'ten_building', (int) $building->id, $building->toArray());
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function buildingsUpdate(Request $request, $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'name' => 'required|string|max:255',
            'address' => 'nullable|string|max:1000',
            'notes' => 'nullable|string|max:2000',
        ]);

        $building = TenBuilding::query()->where('business_id', $businessId)->findOrFail((int) $id);
        $old = $building->toArray();
        $building->update([
            'name' => (string) $request->input('name'),
            'address' => (string) $request->input('address', ''),
            'notes' => (string) $request->input('notes', ''),
        ]);

        MosqueAuditUtil::log($businessId, 'update', 'ten_building', (int) $building->id, ['old' => $old, 'new' => $building->toArray()]);
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function buildingsDestroy($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $building = TenBuilding::query()->where('business_id', $businessId)->findOrFail((int) $id);
        $old = $building->toArray();
        $building->delete();

        MosqueAuditUtil::log($businessId, 'delete', 'ten_building', (int) $building->id, $old);
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function floorsData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $query = DB::table('ten_floors as f')
            ->join('ten_buildings as b', function ($join) use ($businessId) {
                $join->on('b.id', '=', 'f.building_id')
                    ->where('b.business_id', '=', $businessId)
                    ->whereNull('b.deleted_at');
            })
            ->where('f.business_id', $businessId)
            ->whereNull('f.deleted_at')
            ->select(['f.id', 'f.name_or_number', 'f.sort_order', 'b.name as building']);

        $buildingId = (int) $request->input('building_id', 0);
        if ($buildingId > 0) {
            $query->where('f.building_id', $buildingId);
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $edit = '<button class="btn btn-xs btn-primary ten-struct-edit" data-type="floor" data-id="'.(int) $row->id.'"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $deleteUrl = route('mosque.tenancy.structure.floors.destroy', [(int) $row->id], false);
                $delete = '<button data-href="'.$deleteUrl.'" onclick="(function(){var e=(arguments&&arguments[0])?arguments[0]:(window.event||null);if(e){if(e.preventDefault)e.preventDefault();if(e.stopImmediatePropagation)e.stopImmediatePropagation();else if(e.stopPropagation)e.stopPropagation();}var url=\''.$deleteUrl.'\';var ent=\'floor\';if(typeof window.mosqueOpenDeleteConfirm===\'function\'){window.mosqueOpenDeleteConfirm(url,\'DELETE\',ent);return;}window.location.reload();})(); return false;" class="btn btn-xs btn-danger delete_mosque_floor"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function floorsStore(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'building_id' => 'required|integer',
            'name_or_number' => 'required|string|max:120',
            'sort_order' => 'nullable|integer|min:0|max:9999',
        ]);

        $building = TenBuilding::query()->where('business_id', $businessId)->findOrFail((int) $request->input('building_id'));

        $floor = TenFloor::query()->create([
            'business_id' => $businessId,
            'building_id' => (int) $building->id,
            'name_or_number' => (string) $request->input('name_or_number'),
            'sort_order' => (int) $request->input('sort_order', 0),
            'notes' => null,
        ]);

        MosqueAuditUtil::log($businessId, 'create', 'ten_floor', (int) $floor->id, $floor->toArray());
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function floorsUpdate(Request $request, $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'building_id' => 'required|integer',
            'name_or_number' => 'required|string|max:120',
            'sort_order' => 'nullable|integer|min:0|max:9999',
        ]);

        $floor = TenFloor::query()->where('business_id', $businessId)->findOrFail((int) $id);
        $building = TenBuilding::query()->where('business_id', $businessId)->findOrFail((int) $request->input('building_id'));

        $old = $floor->toArray();
        $floor->update([
            'building_id' => (int) $building->id,
            'name_or_number' => (string) $request->input('name_or_number'),
            'sort_order' => (int) $request->input('sort_order', 0),
        ]);

        MosqueAuditUtil::log($businessId, 'update', 'ten_floor', (int) $floor->id, ['old' => $old, 'new' => $floor->toArray()]);
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function floorsDestroy($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $floor = TenFloor::query()->where('business_id', $businessId)->findOrFail((int) $id);
        $old = $floor->toArray();
        $floor->delete();

        MosqueAuditUtil::log($businessId, 'delete', 'ten_floor', (int) $floor->id, $old);
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function roomsData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        try {
            if (! Schema::hasTable('ten_rooms') || ! Schema::hasTable('ten_floors') || ! Schema::hasTable('ten_buildings')) {
                return DataTables::of(collect([]))->make(true);
            }

            $query = DB::table('ten_rooms as r')
                ->join('ten_floors as f', function ($join) use ($businessId) {
                    $join->on('f.id', '=', 'r.floor_id')
                        ->where('f.business_id', '=', $businessId)
                        ->whereNull('f.deleted_at');
                })
                ->join('ten_buildings as b', function ($join) use ($businessId) {
                    $join->on('b.id', '=', 'f.building_id')
                        ->where('b.business_id', '=', $businessId)
                        ->whereNull('b.deleted_at');
                })
                ->where('r.business_id', $businessId)
                ->whereNull('r.deleted_at')
                ->select(['r.id', 'r.code', 'r.type', 'f.name_or_number as floor', 'b.name as building']);

            $buildingId = (int) $request->input('building_id', 0);
            if ($buildingId > 0) {
                $query->where('b.id', $buildingId);
            }
            $floorId = (int) $request->input('floor_id', 0);
            if ($floorId > 0) {
                $query->where('f.id', $floorId);
            }

            return DataTables::of($query)
                ->addColumn('action', function ($row) {
                    $edit = '<button class="btn btn-xs btn-primary ten-struct-edit" data-type="room" data-id="'.(int) $row->id.'"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                    $deleteUrl = route('mosque.tenancy.structure.rooms.destroy', [(int) $row->id], false);
                    $delete = '<button data-href="'.$deleteUrl.'" onclick="(function(){var e=(arguments&&arguments[0])?arguments[0]:(window.event||null);if(e){if(e.preventDefault)e.preventDefault();if(e.stopImmediatePropagation)e.stopImmediatePropagation();else if(e.stopPropagation)e.stopPropagation();}var url=\''.$deleteUrl.'\';var ent=\'room\';if(typeof window.mosqueOpenDeleteConfirm===\'function\'){window.mosqueOpenDeleteConfirm(url,\'DELETE\',ent);return;}window.location.reload();})(); return false;" class="btn btn-xs btn-danger delete_mosque_room"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                    return $edit.' '.$delete;
                })
                ->rawColumns(['action'])
                ->make(true);
        } catch (\Throwable $e) {
            return response()->json([
                'draw' => (int) $request->input('draw', 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
            ]);
        }
    }

    public function roomsStore(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'floor_id' => 'required|integer',
            'code' => 'required|string|max:60',
            'type' => 'required|in:room,apartment,ward',
            'notes' => 'nullable|string|max:2000',
        ], [
            'floor_id.required' => __('mosque::mosque.tenancy_err_floor_required'),
            'code.required' => __('mosque::mosque.tenancy_err_room_required'),
            'type.required' => __('mosque::mosque.tenancy_err_type_required'),
            'type.in' => __('mosque::mosque.tenancy_err_type_required'),
        ]);

        $floor = TenFloor::query()->where('business_id', $businessId)->findOrFail((int) $request->input('floor_id'));

        $room = TenRoom::query()->create([
            'business_id' => $businessId,
            'floor_id' => (int) $floor->id,
            'code' => (string) $request->input('code'),
            'type' => (string) $request->input('type'),
            'capacity_total' => null,
            'notes' => (string) $request->input('notes', ''),
        ]);

        MosqueAuditUtil::log($businessId, 'create', 'ten_room', (int) $room->id, $room->toArray());
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function roomsUpdate(Request $request, $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'floor_id' => 'required|integer',
            'code' => 'required|string|max:60',
            'type' => 'required|in:room,apartment,ward',
            'notes' => 'nullable|string|max:2000',
        ], [
            'floor_id.required' => __('mosque::mosque.tenancy_err_floor_required'),
            'code.required' => __('mosque::mosque.tenancy_err_room_required'),
            'type.required' => __('mosque::mosque.tenancy_err_type_required'),
            'type.in' => __('mosque::mosque.tenancy_err_type_required'),
        ]);

        $room = TenRoom::query()->where('business_id', $businessId)->findOrFail((int) $id);
        $floor = TenFloor::query()->where('business_id', $businessId)->findOrFail((int) $request->input('floor_id'));
        $old = $room->toArray();

        $room->update([
            'floor_id' => (int) $floor->id,
            'code' => (string) $request->input('code'),
            'type' => (string) $request->input('type'),
            'notes' => (string) $request->input('notes', ''),
        ]);

        MosqueAuditUtil::log($businessId, 'update', 'ten_room', (int) $room->id, ['old' => $old, 'new' => $room->toArray()]);
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function roomsDestroy($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $room = TenRoom::query()->where('business_id', $businessId)->findOrFail((int) $id);
        $old = $room->toArray();
        $room->delete();

        MosqueAuditUtil::log($businessId, 'delete', 'ten_room', (int) $room->id, $old);
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function seatsData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $query = DB::table('ten_seats as s')
            ->join('ten_rooms as r', function ($join) use ($businessId) {
                $join->on('r.id', '=', 's.room_id')
                    ->where('r.business_id', '=', $businessId)
                    ->whereNull('r.deleted_at');
            })
            ->join('ten_floors as f', function ($join) use ($businessId) {
                $join->on('f.id', '=', 'r.floor_id')
                    ->where('f.business_id', '=', $businessId)
                    ->whereNull('f.deleted_at');
            })
            ->join('ten_buildings as b', function ($join) use ($businessId) {
                $join->on('b.id', '=', 'f.building_id')
                    ->where('b.business_id', '=', $businessId)
                    ->whereNull('b.deleted_at');
            })
            ->where('s.business_id', $businessId)
            ->whereNull('s.deleted_at')
            ->select(['s.id', 's.code', 's.status', 'r.code as room', 'f.name_or_number as floor', 'b.name as building']);

        $buildingId = (int) $request->input('building_id', 0);
        if ($buildingId > 0) {
            $query->where('b.id', $buildingId);
        }
        $floorId = (int) $request->input('floor_id', 0);
        if ($floorId > 0) {
            $query->where('f.id', $floorId);
        }
        $roomId = (int) $request->input('room_id', 0);
        if ($roomId > 0) {
            $query->where('r.id', $roomId);
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $edit = '<button class="btn btn-xs btn-primary ten-struct-edit" data-type="seat" data-id="'.(int) $row->id.'"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $deleteUrl = route('mosque.tenancy.structure.seats.destroy', [(int) $row->id], false);
                $delete = '<button data-href="'.$deleteUrl.'" onclick="(function(){var e=(arguments&&arguments[0])?arguments[0]:(window.event||null);if(e){if(e.preventDefault)e.preventDefault();if(e.stopImmediatePropagation)e.stopImmediatePropagation();else if(e.stopPropagation)e.stopPropagation();}var url=\''.$deleteUrl.'\';var ent=\'seat\';if(typeof window.mosqueOpenDeleteConfirm===\'function\'){window.mosqueOpenDeleteConfirm(url,\'DELETE\',ent);return;}window.location.reload();})(); return false;" class="btn btn-xs btn-danger delete_mosque_seat"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function seatsStore(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'room_id' => 'required|integer',
            'code' => 'required|string|max:60',
            'status' => 'required|in:available,held,occupied,maintenance',
        ]);

        $room = TenRoom::query()->where('business_id', $businessId)->findOrFail((int) $request->input('room_id'));

        $seat = TenSeat::query()->create([
            'business_id' => $businessId,
            'room_id' => (int) $room->id,
            'code' => (string) $request->input('code'),
            'capacity' => 1,
            'status' => (string) $request->input('status'),
            'leaving_on' => null,
            'notes' => null,
        ]);

        MosqueAuditUtil::log($businessId, 'create', 'ten_seat', (int) $seat->id, $seat->toArray());
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function seatsUpdate(Request $request, $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'room_id' => 'required|integer',
            'code' => 'required|string|max:60',
            'status' => 'required|in:available,held,occupied,maintenance',
        ]);

        $seat = TenSeat::query()->where('business_id', $businessId)->findOrFail((int) $id);
        $room = TenRoom::query()->where('business_id', $businessId)->findOrFail((int) $request->input('room_id'));

        $old = $seat->toArray();
        $seat->update([
            'room_id' => (int) $room->id,
            'code' => (string) $request->input('code'),
            'status' => (string) $request->input('status'),
        ]);

        MosqueAuditUtil::log($businessId, 'update', 'ten_seat', (int) $seat->id, ['old' => $old, 'new' => $seat->toArray()]);
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function seatsDestroy($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $seat = TenSeat::query()->where('business_id', $businessId)->findOrFail((int) $id);
        $old = $seat->toArray();
        $seat->delete();

        MosqueAuditUtil::log($businessId, 'delete', 'ten_seat', (int) $seat->id, $old);
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }
}
