<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Entities\TenCase;
use Modules\Mosque\Entities\TenTenant;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\TenancyKeyUtil;
use Yajra\DataTables\Facades\DataTables;

class TenancyTenantsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if ($businessId <= 0) {
            abort(403, 'Unauthorized action.');
        }
        return $businessId;
    }

    private function ensurePermission(string $perm = 'mosque.tenancy.view'): void
    {
        if (auth()->user()->can('superadmin')) {
            return;
        }
        if (! auth()->user()->can($perm)) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function ensureTenancyEnabled(int $businessId): void
    {
        if (! Schema::hasTable('mosque_settings')) {
            return;
        }
        $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        if (! (bool) ($settings['tenancy_enabled'] ?? true)) {
            abort(403, 'Tenancy is disabled.');
        }
    }

    public function index()
    {
        $this->ensurePermission('mosque.tenancy.view');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        return view('mosque::tenancy.tenants.index');
    }

    public function data(Request $request)
    {
        $this->ensurePermission('mosque.tenancy.view');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $query = TenTenant::query()
            ->where('business_id', $businessId)
            ->select([
                'id',
                'full_name',
                'phone',
                'whatsapp',
                'email',
                'nid_no',
                'passport_no',
                'photo_path',
                'created_at',
            ]);

        return DataTables::of($query)
            ->addColumn('photo', function ($row) {
                $path = (string) ($row->photo_path ?? '');
                if ($path === '') {
                    return '<span class="text-muted">-</span>';
                }
                $url = route('mosque.tenancy.tenants.file', ['id' => (int) $row->id, 'path' => $path], false);
                return '<img src="'.$url.'" style="width:32px;height:32px;border-radius:6px;object-fit:cover;" alt="photo">';
            })
            ->addColumn('action', function ($row) {
                $edit = '';
                if (auth()->user()->can('mosque.tenancy.edit')) {
                    $edit = '<a href="'.route('mosque.tenancy.tenants.edit', [(int) $row->id], false).'" class="btn btn-xs btn-primary"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</a>';
                }
                $delete = '';
                if (auth()->user()->can('mosque.tenancy.edit')) {
                    $deleteUrl = route('mosque.tenancy.tenants.destroy', [(int) $row->id], false);
                    $delete = '<button data-href="'.$deleteUrl.'" onclick="(function(){var e=(arguments&&arguments[0])?arguments[0]:(window.event||null);if(e){if(e.preventDefault)e.preventDefault();if(e.stopImmediatePropagation)e.stopImmediatePropagation();else if(e.stopPropagation)e.stopPropagation();}var url=\''.$deleteUrl.'\';var ent=\'tenant\';if(typeof window.mosqueOpenDeleteConfirm===\'function\'){window.mosqueOpenDeleteConfirm(url,\'DELETE\',ent);return;}window.location.reload();})(); return false;" class="btn btn-xs btn-danger delete_mosque_tenant"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                }
                return trim($edit.' '.$delete);
            })
            ->rawColumns(['photo', 'action'])
            ->make(true);
    }

    public function create()
    {
        $this->ensurePermission('mosque.tenancy.edit');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        return view('mosque::tenancy.tenants.form', ['tenant' => null, 'cases' => [], 'formData' => []]);
    }

    public function edit($id)
    {
        $this->ensurePermission('mosque.tenancy.edit');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $tenant = TenTenant::query()->where('business_id', $businessId)->findOrFail((int) $id);
        return view('mosque::tenancy.tenants.form', ['tenant' => $tenant, 'cases' => [], 'formData' => (array) ($tenant->form_data ?: [])]);
    }

    private function validateSoftUnique(int $businessId, array $data, ?int $ignoreId = null): void
    {
        $phone = trim((string) ($data['phone'] ?? ''));
        if ($phone !== '') {
            $exists = TenTenant::query()
                ->where('business_id', $businessId)
                ->where('phone', $phone)
                ->when($ignoreId, fn ($q) => $q->where('id', '!=', $ignoreId))
                ->exists();
            if ($exists) {
                throw ValidationException::withMessages(['phone' => __('mosque::mosque.tenancy_err_phone_exists')]);
            }
        }

        $nid = trim((string) ($data['nid_no'] ?? ''));
        if ($nid !== '') {
            $exists = TenTenant::query()
                ->where('business_id', $businessId)
                ->where('nid_no', $nid)
                ->when($ignoreId, fn ($q) => $q->where('id', '!=', $ignoreId))
                ->exists();
            if ($exists) {
                throw ValidationException::withMessages(['nid_no' => __('mosque::mosque.tenancy_err_nid_exists')]);
            }
        }
    }

    private function ensureUploadDirs(): void
    {
        try {
            Storage::disk('public')->makeDirectory('mosque_tenancy/tenants');
        } catch (\Throwable $e) {
        }
    }

    private function handleUploads(int $businessId, Request $request, array $payload, ?TenTenant $tenant = null): array
    {
        $this->ensureUploadDirs();

        if ($request->hasFile('photo')) {
            $file = $request->file('photo');
            if ($file && $file->isValid()) {
                $name = 'tenant_'.$businessId.'_'.now()->format('YmdHis').'_'.Str::random(6).'.'.$file->getClientOriginalExtension();
                $path = $file->storeAs('mosque_tenancy/tenants', $name, 'public');
                $payload['photo_path'] = $path;
            }
        }

        $attachments = $tenant?->attachments ?: [];
        if (! is_array($attachments)) {
            $attachments = [];
        }
        if ($request->hasFile('attachments')) {
            foreach ((array) $request->file('attachments') as $file) {
                if (! $file || ! $file->isValid()) {
                    continue;
                }
                $name = 'att_'.$businessId.'_'.now()->format('YmdHis').'_'.Str::random(6).'.'.$file->getClientOriginalExtension();
                $path = $file->storeAs('mosque_tenancy/tenants', $name, 'public');
                $attachments[] = ['name' => $file->getClientOriginalName(), 'path' => $path];
            }
        }
        $payload['attachments'] = $attachments;

        return $payload;
    }

    private function buildFormData(array $existing, array $incoming, array $data): array
    {
        $formData = array_replace_recursive($existing, $incoming);

        $formData['tenant_full_name'] = (string) ($data['full_name'] ?? ($formData['tenant_full_name'] ?? ''));
        $formData['mobile'] = (string) ($data['phone'] ?? ($formData['mobile'] ?? ''));
        $formData['email'] = (string) ($data['email'] ?? ($formData['email'] ?? ''));
        $formData['nid_number'] = (string) (trim((string) ($formData['nid_number'] ?? '')) !== '' ? $formData['nid_number'] : ($data['nid_no'] ?? ''));
        $formData['passport_number'] = (string) (trim((string) ($formData['passport_number'] ?? '')) !== '' ? $formData['passport_number'] : ($data['passport_no'] ?? ''));

        if (! empty($data['dob'])) {
            $formData['dob'] = $data['dob'];
        }
        if (! empty($data['marital_status'])) {
            $formData['marital_status'] = (string) $data['marital_status'];
        }
        if (! empty($data['present_address'])) {
            $formData['present_address'] = (string) $data['present_address'];
        }
        if (! empty($data['permanent_address'])) {
            $formData['permanent_address'] = (string) $data['permanent_address'];
        }

        return $formData;
    }

    private function matchCases(int $businessId, array $data): array
    {
        if (! Schema::hasTable('ten_cases')) {
            return [];
        }
        $keys = TenancyKeyUtil::possibleKeys((string) ($data['phone'] ?? ''), (string) ($data['nid_no'] ?? ''), (string) ($data['passport_no'] ?? ''));
        if (empty($keys)) {
            return [];
        }
        return TenCase::query()
            ->where('business_id', $businessId)
            ->whereIn('tenant_key', $keys)
            ->orderByDesc('id')
            ->limit(20)
            ->get(['id', 'title', 'severity', 'created_at'])
            ->toArray();
    }

    public function store(Request $request)
    {
        $this->ensurePermission('mosque.tenancy.edit');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'full_name' => 'required|string|max:255',
            'phone' => 'required|string|max:50',
            'whatsapp' => 'nullable|string|max:50',
            'email' => 'nullable|email|max:255',
            'home_country' => 'nullable|string|max:80',
            'present_address' => 'nullable|string|max:2000',
            'permanent_address' => 'nullable|string|max:2000',
            'dob' => 'nullable|date',
            'marital_status' => 'nullable|string|max:80',
            'nid_no' => 'nullable|string|max:80',
            'passport_no' => 'nullable|string|max:80',
            'passport_issue_date' => 'nullable|date',
            'passport_expiry_date' => 'nullable|date|after_or_equal:passport_issue_date',
            'form_data' => 'nullable|array',
            'form_data.father_name' => 'required|string|max:255',
            'ack_case_match' => 'nullable|boolean',
            'photo' => 'nullable|file|mimes:jpg,jpeg,png,webp|max:4096',
            'attachments.*' => 'nullable|file|max:5120',
        ]);

        $data = $request->only([
            'full_name',
            'phone',
            'whatsapp',
            'email',
            'home_country',
            'present_address',
            'permanent_address',
            'dob',
            'marital_status',
            'nid_no',
            'passport_no',
            'passport_issue_date',
            'passport_expiry_date',
        ]);

        $incomingFormData = (array) ($request->input('form_data', []));
        $formData = $this->buildFormData([], $incomingFormData, $data);

        if (trim((string) ($data['nid_no'] ?? '')) === '' && trim((string) ($formData['nid_number'] ?? '')) !== '') {
            $data['nid_no'] = (string) $formData['nid_number'];
        }
        if (trim((string) ($data['passport_no'] ?? '')) === '' && trim((string) ($formData['passport_number'] ?? '')) !== '') {
            $data['passport_no'] = (string) $formData['passport_number'];
        }

        if (trim((string) ($data['nid_no'] ?? '')) === '' && trim((string) ($data['passport_no'] ?? '')) === '') {
            throw ValidationException::withMessages(['nid_no' => __('mosque::mosque.tenancy_err_id_required')]);
        }

        $this->validateSoftUnique($businessId, $data, null);

        $cases = $this->matchCases($businessId, $data);
        if (! empty($cases) && ! $request->boolean('ack_case_match')) {
            return response()->json([
                'success' => false,
                'msg' => __('mosque::mosque.tenancy_err_case_ack_required'),
                'cases' => $cases,
            ], 422);
        }

        $payload = array_merge($data, [
            'business_id' => $businessId,
            'user_id' => null,
            'form_data' => $formData,
        ]);

        $payload = $this->handleUploads($businessId, $request, $payload, null);

        $tenant = TenTenant::query()->create($payload);

        MosqueAuditUtil::log($businessId, 'create', 'ten_tenant', (int) $tenant->id, [
            'full_name' => $tenant->full_name,
            'phone' => $tenant->phone,
        ]);

        return ['success' => true, 'msg' => __('lang_v1.success'), 'redirect' => route('mosque.tenancy.tenants', [], false)];
    }

    public function update(Request $request, $id)
    {
        $this->ensurePermission('mosque.tenancy.edit');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $tenant = TenTenant::query()->where('business_id', $businessId)->findOrFail((int) $id);

        $request->validate([
            'full_name' => 'required|string|max:255',
            'phone' => 'required|string|max:50',
            'whatsapp' => 'nullable|string|max:50',
            'email' => 'nullable|email|max:255',
            'home_country' => 'nullable|string|max:80',
            'present_address' => 'nullable|string|max:2000',
            'permanent_address' => 'nullable|string|max:2000',
            'dob' => 'nullable|date',
            'marital_status' => 'nullable|string|max:80',
            'nid_no' => 'nullable|string|max:80',
            'passport_no' => 'nullable|string|max:80',
            'passport_issue_date' => 'nullable|date',
            'passport_expiry_date' => 'nullable|date|after_or_equal:passport_issue_date',
            'form_data' => 'nullable|array',
            'form_data.father_name' => 'nullable|string|max:255',
            'ack_case_match' => 'nullable|boolean',
            'photo' => 'nullable|file|mimes:jpg,jpeg,png,webp|max:4096',
            'attachments.*' => 'nullable|file|max:5120',
        ]);

        $data = $request->only([
            'full_name',
            'phone',
            'whatsapp',
            'email',
            'home_country',
            'present_address',
            'permanent_address',
            'dob',
            'marital_status',
            'nid_no',
            'passport_no',
            'passport_issue_date',
            'passport_expiry_date',
        ]);

        $incomingFormData = (array) ($request->input('form_data', []));
        $formData = $this->buildFormData((array) ($tenant->form_data ?: []), $incomingFormData, $data);

        if (trim((string) ($data['nid_no'] ?? '')) === '' && trim((string) ($formData['nid_number'] ?? '')) !== '') {
            $data['nid_no'] = (string) $formData['nid_number'];
        }
        if (trim((string) ($data['passport_no'] ?? '')) === '' && trim((string) ($formData['passport_number'] ?? '')) !== '') {
            $data['passport_no'] = (string) $formData['passport_number'];
        }

        $effectiveNid = trim((string) (($data['nid_no'] ?? '') !== '' ? $data['nid_no'] : ($tenant->nid_no ?? '')));
        $effectivePassport = trim((string) (($data['passport_no'] ?? '') !== '' ? $data['passport_no'] : ($tenant->passport_no ?? '')));
        $hadAnyId = trim((string) ($tenant->nid_no ?? '')) !== '' || trim((string) ($tenant->passport_no ?? '')) !== '';
        if ($hadAnyId && $effectiveNid === '' && $effectivePassport === '') {
            throw ValidationException::withMessages(['nid_no' => __('mosque::mosque.tenancy_err_id_required')]);
        }

        $this->validateSoftUnique($businessId, $data, (int) $tenant->id);

        $cases = $this->matchCases($businessId, $data);
        if (! empty($cases) && ! $request->boolean('ack_case_match')) {
            return response()->json([
                'success' => false,
                'msg' => __('mosque::mosque.tenancy_err_case_ack_required'),
                'cases' => $cases,
            ], 422);
        }

        $old = $tenant->toArray();
        $payload = array_merge($data, [
            'form_data' => $formData,
        ]);
        $payload = $this->handleUploads($businessId, $request, $payload, $tenant);
        $tenant->update($payload);

        MosqueAuditUtil::log($businessId, 'update', 'ten_tenant', (int) $tenant->id, [
            'old' => $old,
            'new' => $tenant->toArray(),
        ]);

        return ['success' => true, 'msg' => __('lang_v1.success'), 'redirect' => route('mosque.tenancy.tenants', [], false)];
    }

    public function destroy($id)
    {
        $this->ensurePermission('mosque.tenancy.edit');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $tenant = TenTenant::query()->where('business_id', $businessId)->findOrFail((int) $id);
        $old = $tenant->toArray();
        $tenant->delete();

        MosqueAuditUtil::log($businessId, 'delete', 'ten_tenant', (int) $tenant->id, $old);
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function file(Request $request, $id)
    {
        $this->ensurePermission('mosque.tenancy.view');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $tenant = TenTenant::query()->where('business_id', $businessId)->findOrFail((int) $id);
        $path = (string) $request->query('path', '');
        if ($path === '') {
            abort(404);
        }

        $allowed = [];
        if (! empty($tenant->photo_path)) {
            $allowed[] = (string) $tenant->photo_path;
        }
        foreach ((array) ($tenant->attachments ?: []) as $att) {
            if (is_array($att) && ! empty($att['path'])) {
                $allowed[] = (string) $att['path'];
            }
        }

        if (! in_array($path, $allowed, true)) {
            abort(403);
        }

        if (! Storage::disk('public')->exists($path)) {
            abort(404);
        }

        return response()->file(Storage::disk('public')->path($path));
    }
}
