@extends('mosque::layouts.app')
@section('title', __('mosque::mosque.mosque') . ' | Finance Reports')

@section('content')
<section class="content-header">
    <h1 class="tw-text-xl md:tw-text-3xl tw-font-bold tw-text-black">Finance Reports</h1>
    <a href="{{ route('mosque.finance') }}" class="btn btn-default pull-right" style="margin-top:-34px;">
        <i class="fa fa-arrow-left"></i> @lang('messages.back')
    </a>
</section>

<section class="content">
    <style>
        .mf-summary { margin-top: 6px; }
        .mf-box { border-radius: 4px; padding: 10px 12px; font-weight: 700; border: 1px solid rgba(0,0,0,.06); }
        .mf-box small { display:block; font-weight: 800; opacity: .95; }
        .mf-summary .mf-income { background: #e9f7ef; color: #0f5132; }
        .mf-summary .mf-expense { background: #fdecea; color: #842029; }
        .mf-summary .mf-net { background: #e7f1ff; color: #084298; }
        .mf-summary .display_currency { font-weight: 800; }
    </style>

    @component('components.widget', ['class' => 'box-primary'])
        <div class="row">
            <div class="col-sm-3" style="display:none;">
                <div class="form-group">
                    <label>@lang('lang_v1.start_date')</label>
                    <input type="date" id="mfr_start_date" class="form-control" value="{{ now()->startOfMonth()->format('Y-m-d') }}">
                </div>
            </div>
            <div class="col-sm-3" style="display:none;">
                <div class="form-group">
                    <label>@lang('lang_v1.end_date')</label>
                    <input type="date" id="mfr_end_date" class="form-control" value="{{ now()->endOfMonth()->format('Y-m-d') }}">
                </div>
            </div>
            <div class="col-sm-6">
                <div class="form-group">
                    <label>@lang('report.date_range')</label>
                    <input type="text" id="mfr_date_range" class="form-control" readonly autocomplete="off" placeholder="@lang('lang_v1.select_a_date_range')">
                </div>
            </div>
            <div class="col-sm-2">
                <div class="form-group">
                    <label>@lang('lang_v1.source')</label>
                    <select id="mfr_source" class="form-control">
                        <option value="">@lang('lang_v1.all')</option>
                        <option value="module">Module</option>
                        <option value="core_sale">Core Sale</option>
                        <option value="core_purchase">Core Purchase</option>
                    </select>
                </div>
            </div>
            <div class="col-sm-2">
                <div class="form-group">
                    <label>@lang('lang_v1.type')</label>
                    <select id="mfr_type" class="form-control">
                        <option value="">@lang('lang_v1.all')</option>
                        <option value="income">Income</option>
                        <option value="expense">Expense</option>
                    </select>
                </div>
            </div>
            <div class="col-sm-2">
                <div class="form-group">
                    <label>&nbsp;</label>
                    <div class="text-muted" style="padding-top: 8px;">&nbsp;</div>
                </div>
            </div>
        </div>
    @endcomponent

    @component('components.widget', ['class' => 'box-primary'])
        <div class="row mf-summary">
            <div class="col-sm-4">
                <div class="mf-box mf-income">
                    <small>Zakat Income</small>
                    <span id="mfr_zakat_income" class="display_currency" data-currency_symbol="true">0</span>
                </div>
            </div>
            <div class="col-sm-4">
                <div class="mf-box mf-expense">
                    <small>Zakat Expense</small>
                    <span id="mfr_zakat_expense" class="display_currency" data-currency_symbol="true">0</span>
                </div>
            </div>
            <div class="col-sm-4">
                <div class="mf-box mf-net">
                    <small>Zakat Balance</small>
                    <span id="mfr_zakat_balance" class="display_currency" data-currency_symbol="true">0</span>
                </div>
            </div>
        </div>
    @endcomponent

    @component('components.widget', ['class' => 'box-primary'])
        <h4>Category Wise</h4>
        <div class="table-responsive">
            <table class="table table-bordered table-striped" id="mosque_finance_categories_report" style="width:100%;">
                <thead>
                    <tr>
                        <th>Source</th>
                        <th>Type</th>
                        <th>Category</th>
                        <th>Total</th>
                    </tr>
                </thead>
            </table>
        </div>
        <div class="text-right text-muted" id="mosque_finance_categories_report_count" style="margin-top:6px;"></div>
    @endcomponent

    @component('components.widget', ['class' => 'box-primary'])
        <h4>Monthly Income vs Expense</h4>
        <div class="table-responsive">
            <table class="table table-bordered table-striped" id="mosque_finance_monthly_report" style="width:100%;">
                <thead>
                    <tr>
                        <th>Month</th>
                        <th>Income</th>
                        <th>Expense</th>
                        <th>Net</th>
                    </tr>
                </thead>
                <tbody></tbody>
            </table>
        </div>
        <div class="text-right text-muted" id="mosque_finance_monthly_report_count" style="margin-top:6px;"></div>
    @endcomponent

    @component('components.widget', ['class' => 'box-primary'])
        <h4>Weekly Income vs Expense</h4>
        <div class="table-responsive">
            <table class="table table-bordered table-striped" id="mosque_finance_weekly_report" style="width:100%;">
                <thead>
                    <tr>
                        <th>Week</th>
                        <th>Income</th>
                        <th>Expense</th>
                        <th>Net</th>
                    </tr>
                </thead>
                <tbody></tbody>
            </table>
        </div>
        <div class="text-right text-muted" id="mosque_finance_weekly_report_count" style="margin-top:6px;"></div>
    @endcomponent
</section>
@endsection

@section('javascript')
<script type="text/javascript">
$(document).ready(function () {
    // Force defaults on reload (avoid browser restoring old values).
    function resetFiltersToDefault() {
        var s = moment().startOf('year');
        var e = moment().endOf('year');
        $('#mfr_start_date').val(s.format('YYYY-MM-DD'));
        $('#mfr_end_date').val(e.format('YYYY-MM-DD'));
        $('#mfr_source').val('');
        $('#mfr_type').val('');
        if ($('#mfr_date_range').length && typeof moment_date_format !== 'undefined') {
            $('#mfr_date_range').val(s.format(moment_date_format) + ' ~ ' + e.format(moment_date_format));
        }
    }
    resetFiltersToDefault();

    // Date range filter (like Finance table)
    if (typeof dateRangeSettings !== 'undefined' && $('#mfr_date_range').length) {
        var start = moment($('#mfr_start_date').val(), 'YYYY-MM-DD');
        var end = moment($('#mfr_end_date').val(), 'YYYY-MM-DD');
        $('#mfr_date_range').daterangepicker(
            $.extend({}, dateRangeSettings, { startDate: start, endDate: end }),
            function (s, e) {
                $('#mfr_start_date').val(s.format('YYYY-MM-DD'));
                $('#mfr_end_date').val(e.format('YYYY-MM-DD'));
                $('#mfr_date_range').val(s.format(moment_date_format) + ' ~ ' + e.format(moment_date_format));
                catTable.ajax.reload();
                refreshMonthly();
                refreshWeekly();
                refreshZakat();
            }
        );
        $('#mfr_date_range').val(start.format(moment_date_format) + ' ~ ' + end.format(moment_date_format));
        $('#mfr_date_range').on('cancel.daterangepicker', function () {
            resetFiltersToDefault();
            if ($('#mfr_date_range').data('daterangepicker')) {
                var s = moment($('#mfr_start_date').val(), 'YYYY-MM-DD');
                var e = moment($('#mfr_end_date').val(), 'YYYY-MM-DD');
                $('#mfr_date_range').data('daterangepicker').setStartDate(s);
                $('#mfr_date_range').data('daterangepicker').setEndDate(e);
            }
            catTable.ajax.reload();
            refreshMonthly();
            refreshWeekly();
            refreshZakat();
        });
    }

    function getFilters() {
        return {
            start_date: $('#mfr_start_date').val(),
            end_date: $('#mfr_end_date').val(),
            source: $('#mfr_source').val(),
            type: $('#mfr_type').val(),
        };
    }

    function refreshZakat() {
        $.ajax({
            method: 'GET',
            url: '{{ route("mosque.finance.reports.zakat") }}',
            data: { start_date: $('#mfr_start_date').val(), end_date: $('#mfr_end_date').val() },
            success: function (result) {
                if (result && result.success) {
                    $('#mfr_zakat_income').attr('data-orig-value', result.income).text(result.income);
                    $('#mfr_zakat_expense').attr('data-orig-value', result.expense).text(result.expense);
                    $('#mfr_zakat_balance').attr('data-orig-value', result.balance).text(result.balance);
                    __currency_convert_recursively($('.content'));
                }
            }
        });
    }

    var catTable = $('#mosque_finance_categories_report').DataTable({
        processing: true,
        serverSide: true,
        ajax: {
            url: '{{ route("mosque.finance.reports.categories") }}',
            data: function (d) {
                return $.extend({}, d, getFilters());
            }
        },
        order: [[1, 'asc']],
        dom: 'lBfrtip',
        buttons: [
            { extend: 'csv', text: '<i class="fas fa-file-csv"></i>', className: 'btn btn-success' },
            { extend: 'excel', text: '<i class="fas fa-file-excel"></i>', className: 'btn btn-info' },
            { extend: 'pdf', text: '<i class="fas fa-file-pdf"></i>', className: 'btn btn-danger' },
            { extend: 'print', text: '<i class="fas fa-print"></i>', className: 'btn btn-warning' },
            { extend: 'colvis', text: '<i class="fas fa-eye"></i>', className: 'btn btn-primary' }
        ],
        columns: [
            { data: 'source', name: 'source' },
            { data: 'type', name: 'type' },
            { data: 'category', name: 'category' },
            { data: 'total', name: 'total', orderable: false, searchable: false },
        ],
        drawCallback: function () {
            __currency_convert_recursively($('#mosque_finance_categories_report'));
            var info = catTable ? catTable.page.info() : null;
            if (info) {
                $('#mosque_finance_categories_report_count').text('Total rows: ' + info.recordsDisplay);
            }
        }
    });

    catTable.on('init.dt', function () {
        $('#mosque_finance_categories_report_wrapper .dt-buttons').addClass('justify-content-center flex-wrap mb-2');
    });

    function refreshMonthly() {
        $.ajax({
            method: 'GET',
            url: '{{ route("mosque.finance.reports.monthly") }}',
            data: getFilters(),
            success: function (result) {
                if (!result || !result.success) return;
                var $tbody = $('#mosque_finance_monthly_report tbody');
                $tbody.empty();

                (result.rows || []).forEach(function (row) {
                    var income = parseFloat(row.income || 0);
                    var expense = parseFloat(row.expense || 0);
                    var net = income - expense;
                    var monthLabel = row.ym;
                    if (typeof moment !== 'undefined' && row.ym) {
                        var m = moment(row.ym + '-01', 'YYYY-MM-DD');
                        if (m.isValid()) {
                            monthLabel = m.format('MMMM, YYYY');
                        }
                    }
                    $tbody.append(
                        '<tr>' +
                            '<td>' + monthLabel + '</td>' +
                            '<td><span class="display_currency" data-currency_symbol="true" data-orig-value="' + income + '">' + income + '</span></td>' +
                            '<td><span class="display_currency" data-currency_symbol="true" data-orig-value="' + expense + '">' + expense + '</span></td>' +
                            '<td><span class="display_currency" data-currency_symbol="true" data-orig-value="' + net + '">' + net + '</span></td>' +
                        '</tr>'
                    );
                });
                __currency_convert_recursively($('#mosque_finance_monthly_report'));
                $('#mosque_finance_monthly_report_count').text('Total rows: ' + ((result.rows || []).length));
            }
        });
    }

    function refreshWeekly() {
        $.ajax({
            method: 'GET',
            url: '{{ route("mosque.finance.reports.weekly") }}',
            data: getFilters(),
            success: function (result) {
                if (!result || !result.success) return;
                var $tbody = $('#mosque_finance_weekly_report tbody');
                $tbody.empty();

                (result.rows || []).forEach(function (row) {
                    var income = parseFloat(row.income || 0);
                    var expense = parseFloat(row.expense || 0);
                    var net = income - expense;

                    var label = (row.week_start || '') + ' - ' + (row.week_end || '');
                    if (typeof moment !== 'undefined' && row.week_start && row.week_end) {
                        var ws = moment(row.week_start, 'YYYY-MM-DD');
                        var we = moment(row.week_end, 'YYYY-MM-DD');
                        if (ws.isValid() && we.isValid()) {
                            label = ws.format('DD MMM') + ' – ' + we.format('DD MMM YYYY');
                        }
                    }

                    $tbody.append(
                        '<tr>' +
                            '<td>' + label + '</td>' +
                            '<td><span class="display_currency" data-currency_symbol="true" data-orig-value="' + income + '">' + income + '</span></td>' +
                            '<td><span class="display_currency" data-currency_symbol="true" data-orig-value="' + expense + '">' + expense + '</span></td>' +
                            '<td><span class="display_currency" data-currency_symbol="true" data-orig-value="' + net + '">' + net + '</span></td>' +
                        '</tr>'
                    );
                });
                __currency_convert_recursively($('#mosque_finance_weekly_report'));
                $('#mosque_finance_weekly_report_count').text('Total rows: ' + ((result.rows || []).length));
            }
        });
    }

    $(document).on('change', '#mfr_start_date, #mfr_end_date, #mfr_source, #mfr_type', function () {
        catTable.ajax.reload();
        refreshMonthly();
        refreshWeekly();
        refreshZakat();
    });

    window.addEventListener('pageshow', function () {
        resetFiltersToDefault();
        if ($('#mfr_date_range').data('daterangepicker')) {
            var s = moment($('#mfr_start_date').val(), 'YYYY-MM-DD');
            var e = moment($('#mfr_end_date').val(), 'YYYY-MM-DD');
            $('#mfr_date_range').data('daterangepicker').setStartDate(s);
            $('#mfr_date_range').data('daterangepicker').setEndDate(e);
            $('#mfr_date_range').val(s.format(moment_date_format) + ' ~ ' + e.format(moment_date_format));
        }
        catTable.ajax.reload();
        refreshMonthly();
        refreshWeekly();
        refreshZakat();
    });

    refreshMonthly();
    refreshWeekly();
    refreshZakat();
});
</script>
@endsection
