<?php

namespace Modules\Mosque\Services;

use Illuminate\Support\Facades\DB;
use Modules\Mosque\Entities\MosqueFinanceCategory;
use Modules\Mosque\Entities\MosqueFinanceEntry;
use Modules\Mosque\Entities\MosqueMemberFee;
use Modules\Mosque\Entities\MosqueMemberPayment;

class MemberFeeService
{
    public function cancelFee(int $businessId, int $feeId, int $userId, string $reason): MosqueMemberFee
    {
        return DB::transaction(function () use ($businessId, $feeId, $userId, $reason) {
            $fee = MosqueMemberFee::query()
                ->where('business_id', $businessId)
                ->lockForUpdate()
                ->findOrFail($feeId);

            if ((string) ($fee->lifecycle_status ?? '') === 'cancelled') {
                abort(422, 'Already cancelled.');
            }

            $paymentIds = MosqueMemberPayment::query()
                ->where('business_id', $businessId)
                ->where('member_fee_id', $fee->id)
                ->pluck('id')
                ->map(fn ($id) => (int) $id)
                ->all();

            if (! empty($paymentIds)) {
                MosqueMemberPayment::query()
                    ->where('business_id', $businessId)
                    ->whereIn('id', $paymentIds)
                    ->update([
                        'voided_at' => now(),
                        'voided_by' => $userId,
                    ]);

                MosqueFinanceEntry::query()
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'membership')
                    ->whereIn('ref_id', $paymentIds)
                    ->delete();
            }

            $fee->lifecycle_status = 'cancelled';
            $fee->cancel_reason = $reason;
            $fee->cancelled_by = $userId ?: null;
            $fee->cancelled_at = now();
            $fee->save();

            return $fee;
        });
    }

    public function reactivateFee(int $businessId, int $feeId, int $userId): MosqueMemberFee
    {
        return DB::transaction(function () use ($businessId, $feeId, $userId) {
            $fee = MosqueMemberFee::query()
                ->where('business_id', $businessId)
                ->lockForUpdate()
                ->findOrFail($feeId);

            if ((string) ($fee->lifecycle_status ?? '') !== 'cancelled') {
                abort(422, 'Only cancelled fees can be reactivated.');
            }

            $payments = MosqueMemberPayment::query()
                ->where('business_id', $businessId)
                ->where('member_fee_id', $fee->id)
                ->orderBy('id')
                ->get();

            $paymentIds = $payments->pluck('id')->map(fn ($id) => (int) $id)->all();

            if (! empty($paymentIds)) {
                MosqueMemberPayment::query()
                    ->where('business_id', $businessId)
                    ->whereIn('id', $paymentIds)
                    ->update([
                        'voided_at' => null,
                        'voided_by' => null,
                    ]);

                MosqueFinanceEntry::withTrashed()
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'membership')
                    ->whereIn('ref_id', $paymentIds)
                    ->restore();

                $existingEntryPaymentIds = MosqueFinanceEntry::query()
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'membership')
                    ->whereIn('ref_id', $paymentIds)
                    ->pluck('ref_id')
                    ->map(fn ($id) => (int) $id)
                    ->all();

                $missing = array_values(array_diff($paymentIds, $existingEntryPaymentIds));
                if (! empty($missing)) {
                    $category = MosqueFinanceCategory::query()->firstOrCreate(
                        ['business_id' => $businessId, 'type' => 'income', 'name' => 'Membership fees'],
                        ['active' => true, 'sort_order' => 2]
                    );

                    foreach ($payments as $payment) {
                        if (! in_array((int) $payment->id, $missing, true)) {
                            continue;
                        }
                        MosqueFinanceEntry::query()->create([
                            'business_id' => $businessId,
                            'location_id' => null,
                            'type' => 'income',
                            'category_id' => $category->id,
                            'amount' => $payment->amount,
                            'entry_date' => $payment->paid_on,
                            'ref_module' => 'membership',
                            'ref_id' => $payment->id,
                            'fund_tag' => null,
                            'note' => $payment->ref_no,
                            'created_by' => $userId ?: null,
                        ]);
                    }
                }
            }

            $paidAmount = (float) MosqueMemberPayment::query()
                ->where('business_id', $businessId)
                ->where('member_fee_id', $fee->id)
                ->sum('amount');

            $fee->paid_amount = $paidAmount;
            if ($paidAmount >= (float) $fee->due_amount) {
                $fee->status = 'paid';
                $fee->lifecycle_status = 'paid';
            } elseif ($paidAmount > 0) {
                $fee->status = 'partial';
                $fee->lifecycle_status = 'due';
            } else {
                $fee->status = 'pending';
                $fee->lifecycle_status = 'due';
            }

            $fee->cancel_reason = null;
            $fee->cancelled_by = null;
            $fee->cancelled_at = null;
            $fee->save();

            return $fee;
        });
    }

    public function deleteCancelledFee(int $businessId, int $feeId): void
    {
        DB::transaction(function () use ($businessId, $feeId) {
            $fee = MosqueMemberFee::query()
                ->where('business_id', $businessId)
                ->lockForUpdate()
                ->findOrFail($feeId);

            if ((string) ($fee->lifecycle_status ?? '') !== 'cancelled') {
                abort(422, 'Cancel first.');
            }

            $paymentIds = MosqueMemberPayment::query()
                ->where('business_id', $businessId)
                ->where('member_fee_id', $fee->id)
                ->pluck('id')
                ->map(fn ($id) => (int) $id)
                ->all();

            if (! empty($paymentIds)) {
                MosqueFinanceEntry::withTrashed()
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'membership')
                    ->whereIn('ref_id', $paymentIds)
                    ->forceDelete();

                MosqueMemberPayment::query()
                    ->where('business_id', $businessId)
                    ->whereIn('id', $paymentIds)
                    ->delete();
            }

            $fee->delete();
        });
    }
}

