<?php

namespace Modules\Mosque\Services;

use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class PledgeStatusService
{
    public static function recalcPledge(int $businessId, int $pledgeId, array $settings = []): array
    {
        $pledge = DB::table('mosq_pledges')
            ->where('business_id', $businessId)
            ->where('id', $pledgeId)
            ->first();

        if (! $pledge) {
            return ['est_total' => 0.0, 'fulfilled' => 0.0, 'balance' => 0.0, 'status' => 'draft'];
        }

        $estTotal = (float) ($pledge->est_total_value ?? 0);

        $fulfilledCash = (float) DB::table('mosq_pledge_fulfillments')
            ->where('pledge_id', $pledgeId)
            ->whereIn('channel', ['cash', 'bank', 'card', 'online', 'mfs'])
            ->sum(DB::raw('COALESCE(amount_cash,0)'));

        $fulfilledInKind = (float) DB::table('mosq_pledge_fulfillments')
            ->where('pledge_id', $pledgeId)
            ->where('channel', 'in_kind')
            ->sum(DB::raw('COALESCE(est_value,0)'));

        $fulfilled = $fulfilledCash + $fulfilledInKind;
        $balance = max($estTotal - $fulfilled, 0);

        $currentStatus = (string) ($pledge->status ?? 'draft');
        $newStatus = $currentStatus;
        if (! in_array($currentStatus, ['cancelled'], true)) {
            $tiny = 0.01;
            if ($fulfilled >= ($estTotal - $tiny) && $estTotal > 0) {
                $newStatus = 'fulfilled';
            } elseif ($fulfilled > $tiny) {
                $newStatus = 'partially_fulfilled';
            } else {
                $dueDate = ! empty($pledge->due_date) ? Carbon::parse($pledge->due_date) : null;
                if ($currentStatus === 'draft') {
                    $newStatus = 'draft';
                } elseif (! $dueDate) {
                    $newStatus = 'active';
                } elseif ($dueDate->isPast()) {
                    $newStatus = 'overdue';
                } else {
                    $windows = self::parseWindows((string) ($settings['pledges_reminder_windows'] ?? '7,30'));
                    $soonDays = ! empty($windows) ? min($windows) : 7;
                    $diff = Carbon::today()->diffInDays($dueDate, false);
                    $newStatus = ($diff >= 0 && $diff <= $soonDays) ? 'due_soon' : 'active';
                }
            }
        }

        DB::table('mosq_pledges')
            ->where('business_id', $businessId)
            ->where('id', $pledgeId)
            ->update([
                'status' => $newStatus,
                'updated_at' => now(),
            ]);

        return [
            'est_total' => $estTotal,
            'fulfilled' => $fulfilled,
            'balance' => $balance,
            'status' => $newStatus,
        ];
    }

    public static function recalcBusiness(int $businessId, array $settings = []): void
    {
        $ids = DB::table('mosq_pledges')
            ->where('business_id', $businessId)
            ->whereNotIn('status', ['cancelled'])
            ->pluck('id');

        foreach ($ids as $id) {
            self::recalcPledge($businessId, (int) $id, $settings);
        }
    }

    private static function parseWindows(string $raw): array
    {
        $parts = array_filter(array_map('trim', explode(',', $raw)), fn ($x) => $x !== '');
        $nums = [];
        foreach ($parts as $p) {
            if (is_numeric($p)) {
                $nums[] = (int) $p;
            }
        }
        $nums = array_values(array_unique(array_filter($nums, fn ($n) => $n > 0)));
        sort($nums);
        return $nums;
    }
}

