<?php

namespace Modules\Mosque\Services;

use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\TenAllocation;
use Modules\Mosque\Entities\TenBill;
use Modules\Mosque\Entities\TenTenant;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\TenancyNoticeUtil;

class TenancyBillingService
{
    /**
     * Generate due bills up to today (idempotent).
     * Options:
     * - max_allocations: int
     * - send_notices: bool
     */
    public static function generateDueBills(int $businessId, array $options = []): array
    {
        if ($businessId <= 0 || ! Schema::hasTable('ten_allocations') || ! Schema::hasTable('ten_bills')) {
            return ['generated' => 0, 'skipped' => 0, 'whatsapp_links' => []];
        }

        $max = (int) ($options['max_allocations'] ?? 200);
        $sendNotices = (bool) ($options['send_notices'] ?? true);

        $today = Carbon::today();
        $generated = 0;
        $skipped = 0;
        $waLinks = [];

        $allocations = TenAllocation::query()
            ->where('business_id', $businessId)
            ->where('status', 'active')
            ->whereDate('next_bill_on', '<=', $today->toDateString())
            ->orderBy('next_bill_on')
            ->limit(max(1, $max))
            ->get();

        foreach ($allocations as $allocation) {
            try {
                $periodFrom = Carbon::parse((string) $allocation->next_bill_on)->startOfDay();
                if ($periodFrom->gt($today)) {
                    $skipped++;
                    continue;
                }

                $cycle = (string) ($allocation->billing_cycle ?? 'monthly');
                $periodTo = $cycle === 'yearly'
                    ? $periodFrom->copy()->addYear()->subDay()
                    : $periodFrom->copy()->addMonth()->subDay();

                $nextBillOn = $cycle === 'yearly'
                    ? $periodFrom->copy()->addYear()
                    : $periodFrom->copy()->addMonth();

                $bill = null;
                DB::transaction(function () use ($businessId, $allocation, $periodFrom, $periodTo, $nextBillOn, &$bill) {
                    // Idempotency: enforce unique constraint and also query for existing.
                    $existing = TenBill::query()
                        ->where('business_id', $businessId)
                        ->where('allocation_id', $allocation->id)
                        ->whereDate('period_from', $periodFrom->toDateString())
                        ->whereDate('period_to', $periodTo->toDateString())
                        ->first();

                    if ($existing) {
                        $bill = $existing;
                        return;
                    }

                    $bill = TenBill::query()->create([
                        'business_id' => $businessId,
                        'allocation_id' => (int) $allocation->id,
                        'period_from' => $periodFrom->toDateString(),
                        'period_to' => $periodTo->toDateString(),
                        'amount' => (float) ($allocation->rent_amount ?? 0),
                        'tax' => null,
                        'discount' => null,
                        'status' => 'unpaid',
                        'pos_invoice_id' => null,
                        'generated_on' => Carbon::today()->toDateString(),
                    ]);

                    $allocation->next_bill_on = $nextBillOn->toDateString();
                    $allocation->save();
                });

                if (! $bill) {
                    $skipped++;
                    continue;
                }

                $generated++;
                MosqueAuditUtil::log($businessId, 'generate', 'ten_bill', (int) $bill->id, [
                    'allocation_id' => (int) $allocation->id,
                    'period_from' => (string) $bill->period_from,
                    'period_to' => (string) $bill->period_to,
                    'amount' => (float) $bill->amount,
                ]);

                if ($sendNotices) {
                    $tenant = TenTenant::query()
                        ->where('business_id', $businessId)
                        ->find((int) $allocation->tenant_id);
                    if ($tenant) {
                        $notify = TenancyNoticeUtil::billGenerated($businessId, $tenant, $bill);
                        if (! empty($notify['whatsapp_links'])) {
                            $waLinks = array_merge($waLinks, (array) $notify['whatsapp_links']);
                        }
                    }
                }
            } catch (\Throwable $e) {
                $skipped++;
            }
        }

        return [
            'generated' => $generated,
            'skipped' => $skipped,
            'whatsapp_links' => $waLinks,
        ];
    }

    public static function previewDueBills(int $businessId, int $limit = 10): array
    {
        if ($businessId <= 0 || ! Schema::hasTable('ten_allocations')) {
            return ['due' => []];
        }

        $today = Carbon::today()->toDateString();

        $rows = DB::table('ten_allocations as a')
            ->join('ten_seats as s', function ($join) use ($businessId) {
                $join->on('s.id', '=', 'a.seat_id')
                    ->where('s.business_id', '=', $businessId)
                    ->whereNull('s.deleted_at');
            })
            ->join('ten_rooms as r', function ($join) use ($businessId) {
                $join->on('r.id', '=', 's.room_id')
                    ->where('r.business_id', '=', $businessId)
                    ->whereNull('r.deleted_at');
            })
            ->join('ten_floors as f', function ($join) use ($businessId) {
                $join->on('f.id', '=', 'r.floor_id')
                    ->where('f.business_id', '=', $businessId)
                    ->whereNull('f.deleted_at');
            })
            ->join('ten_buildings as b', function ($join) use ($businessId) {
                $join->on('b.id', '=', 'f.building_id')
                    ->where('b.business_id', '=', $businessId)
                    ->whereNull('b.deleted_at');
            })
            ->join('ten_tenants as t', function ($join) use ($businessId) {
                $join->on('t.id', '=', 'a.tenant_id')
                    ->where('t.business_id', '=', $businessId)
                    ->whereNull('t.deleted_at');
            })
            ->where('a.business_id', $businessId)
            ->whereNull('a.deleted_at')
            ->where('a.status', 'active')
            ->whereDate('a.next_bill_on', '<=', $today)
            ->orderBy('a.next_bill_on')
            ->limit(max(1, $limit))
            ->select([
                'a.id',
                'a.next_bill_on',
                'a.rent_amount',
                'a.billing_cycle',
                't.full_name as tenant',
                DB::raw("CONCAT(b.name,' / ',f.name_or_number,' / ',r.code,' / ',s.code) as seat_label"),
            ])
            ->get();

        return ['due' => $rows];
    }
}

