<?php

namespace Modules\Mosque\Utils;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class MosqueBankBalanceUtil
{
    public static function bankWalletTotal(int $businessId, ?string $asOfDate = null): float
    {
        if (! Schema::hasTable('mosque_bank_wallet_transactions')) {
            return 0.0;
        }

        $creditQ = DB::table('mosque_bank_wallet_transactions')
            ->where('business_id', $businessId)
            ->where('direction', 'credit');
        if (! empty($asOfDate)) {
            $creditQ->whereDate('txn_date', '<=', $asOfDate);
        }
        $credit = (float) $creditQ->sum('amount');

        $debitQ = DB::table('mosque_bank_wallet_transactions')
            ->where('business_id', $businessId)
            ->where('direction', 'debit');
        if (! empty($asOfDate)) {
            $debitQ->whereDate('txn_date', '<=', $asOfDate);
        }
        $debit = (float) $debitQ->sum('amount');

        return (float) ($credit - $debit);
    }

    public static function bankWalletBalanceByAccount(int $businessId, int $bankAccountId, ?string $asOfDate = null): float
    {
        if (! Schema::hasTable('mosque_bank_wallet_transactions')) {
            return 0.0;
        }

        $creditQ = DB::table('mosque_bank_wallet_transactions')
            ->where('business_id', $businessId)
            ->where('bank_account_id', $bankAccountId)
            ->where('direction', 'credit');
        if (! empty($asOfDate)) {
            $creditQ->whereDate('txn_date', '<=', $asOfDate);
        }
        $credit = (float) $creditQ->sum('amount');

        $debitQ = DB::table('mosque_bank_wallet_transactions')
            ->where('business_id', $businessId)
            ->where('bank_account_id', $bankAccountId)
            ->where('direction', 'debit');
        if (! empty($asOfDate)) {
            $debitQ->whereDate('txn_date', '<=', $asOfDate);
        }
        $debit = (float) $debitQ->sum('amount');

        return (float) ($credit - $debit);
    }

    public static function overallNetBalance(int $businessId, ?string $asOfDate = null): float
    {
        if (! Schema::hasTable('mosque_finance_entries')) {
            return 0.0;
        }

        $incomeQ = DB::table('mosque_finance_entries')
            ->where('business_id', $businessId)
            ->where('type', 'income')
            ->whereNull('deleted_at');
        if (! empty($asOfDate)) {
            $incomeQ->whereDate('entry_date', '<=', $asOfDate);
        }
        $income = (float) $incomeQ->sum('amount');

        $expenseQ = DB::table('mosque_finance_entries')
            ->where('business_id', $businessId)
            ->where('type', 'expense')
            ->whereNull('deleted_at');
        if (! empty($asOfDate)) {
            $expenseQ->whereDate('entry_date', '<=', $asOfDate);
        }
        $expense = (float) $expenseQ->sum('amount');

        return (float) ($income - $expense);
    }

    /**
     * Cash balance formula:
     * cash_balance = overall_net_balance - total_bank_wallet_balance
     */
    public static function cashNetBalance(int $businessId, ?string $asOfDate = null): float
    {
        $overall = self::overallNetBalance($businessId, $asOfDate);
        $bankTotal = self::bankWalletTotal($businessId, $asOfDate);
        return (float) ($overall - $bankTotal);
    }
}
