<?php

namespace Modules\Mosque\Utils;

use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueCommittee;
use Modules\Mosque\Entities\MosqueCommitteeMember;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueSetting;

class MosqueDeleteNotificationUtil
{
    public static function notify(int $businessId, string $entity, ?int $entityId = null, array $details = []): array
    {
        $entity = trim($entity);
        if ($entity === '') {
            $entity = 'item';
        }

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        }

        $recipients = self::committeeRecipients($businessId);

        $businessName = (string) request()->session()->get('business.name')
            ?: (string) request()->session()->get('business.business_name')
            ?: 'Business';

        $mosqueName = ! empty($profile?->name) ? (string) $profile->name : $businessName;
        $actor = auth()->check() ? (string) (auth()->user()->first_name.' '.auth()->user()->last_name) : 'System';
        $actor = trim($actor) === '' ? 'System' : $actor;

        $detailsText = trim(json_encode($details, JSON_PRETTY_PRINT));
        if ($detailsText === '' || $detailsText === '[]' || $detailsText === 'null') {
            $detailsText = '';
        }

        $tokens = [
            '{business}' => $businessName,
            '{mosque}' => $mosqueName,
            '{actor}' => $actor,
            '{entity}' => $entity,
            '{entity_id}' => (string) ($entityId ?? ''),
            '{date}' => now()->format('Y-m-d H:i'),
            '{details}' => $detailsText,
        ];

        $whatsappTemplate = (string) ($settings['template_delete_whatsapp_message'] ?? '');
        if (trim($whatsappTemplate) === '') {
            $whatsappTemplate = "Assalamu Alaikum,\n{actor} deleted {entity} #{entity_id}\n{mosque}\n{date}";
        }

        $emailSubjectTemplate = (string) ($settings['template_delete_email_subject'] ?? '');
        if (trim($emailSubjectTemplate) === '') {
            $emailSubjectTemplate = "Delete Notification: {entity} #{entity_id}";
        }

        $emailBodyTemplate = (string) ($settings['template_delete_email_message'] ?? '');
        if (trim($emailBodyTemplate) === '') {
            $emailBodyTemplate = "Assalamu Alaikum,<br><br>{actor} deleted <strong>{entity}</strong> #{entity_id}.<br><br><strong>Mosque:</strong> {mosque}<br><strong>Business:</strong> {business}<br><strong>Date:</strong> {date}<br><br><pre style=\"white-space:pre-wrap;\">{details}</pre>";
        }

        $whatsappBody = strtr($whatsappTemplate, $tokens);
        $emailSubject = strtr($emailSubjectTemplate, $tokens);
        $emailBody = strtr($emailBodyTemplate, array_merge($tokens, [
            '{details}' => e($detailsText),
        ]));

        $emailSent = 0;
        $emailFailed = 0;
        $whatsappLinks = [];

        foreach ($recipients as $r) {
            $email = (string) ($r['email'] ?? '');
            $phone = (string) ($r['phone'] ?? '');
            $name = (string) ($r['name'] ?? 'Member');

            if ($email !== '') {
                try {
                    Mail::send([], [], function ($message) use ($email, $emailSubject, $emailBody) {
                        $message->to($email)
                            ->subject($emailSubject)
                            ->setBody($emailBody, 'text/html');
                    });
                    $emailSent++;
                } catch (\Exception $e) {
                    $emailFailed++;
                }
            }

            $digits = preg_replace('/\\D+/', '', $phone);
            if (! empty($digits)) {
                $whatsappLinks[] = [
                    'name' => $name.' ('.$digits.')',
                    'url' => 'https://wa.me/'.$digits.'?text='.rawurlencode($whatsappBody),
                ];
            }
        }

        return [
            'emails_sent' => $emailSent,
            'emails_failed' => $emailFailed,
            'whatsapp_links' => $whatsappLinks,
        ];
    }

    private static function committeeRecipients(int $businessId): array
    {
        if (! Schema::hasTable('mosque_committees') || ! Schema::hasTable('mosque_committee_members')) {
            return [];
        }

        $committee = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->orderByDesc('id')
            ->first();

        $query = MosqueCommitteeMember::query()->where('business_id', $businessId);
        if (! empty($committee)) {
            $query->where('committee_id', $committee->id);
        }

        $members = $query->get();

        $unique = [];
        foreach ($members as $m) {
            $email = strtolower(trim((string) ($m->email ?? '')));
            $phone = trim((string) ($m->phone ?? ''));
            $key = $email !== '' ? 'e:'.$email : ($phone !== '' ? 'p:'.preg_replace('/\\D+/', '', $phone) : null);
            if (empty($key)) {
                continue;
            }
            if (isset($unique[$key])) {
                continue;
            }
            $unique[$key] = [
                'name' => (string) ($m->member_name ?? 'Member'),
                'email' => (string) ($m->email ?? ''),
                'phone' => (string) ($m->phone ?? ''),
            ];
        }

        return array_values($unique);
    }
}

