<?php

namespace Modules\Mosque\Utils;

use App\Currency;
use Illuminate\Support\Facades\Storage;
use Modules\Mosque\Entities\MosqueProfile;

class MosqueDocumentUtil
{
    public static function logoDataUri(?MosqueProfile $profile): ?string
    {
        if (empty($profile) || empty($profile->logo)) {
            return null;
        }

        try {
            $relativePath = 'mosque_logos/'.basename((string) $profile->logo);
            $diskDefault = (string) config('filesystems.default');
            $diskCandidates = array_values(array_unique([$diskDefault, 'public']));

            $path = null;
            foreach ($diskCandidates as $disk) {
                try {
                    if (Storage::disk($disk)->exists($relativePath)) {
                        $path = Storage::disk($disk)->path($relativePath);
                        break;
                    }
                } catch (\Throwable $e) {
                    // ignore and try next disk
                }
            }

            if (! is_string($path) || $path === '' || ! file_exists($path)) {
                return null;
            }

            $mime = @mime_content_type($path) ?: 'image/png';
            $content = @file_get_contents($path);
            if ($content === false) {
                return null;
            }

            return 'data:'.$mime.';base64,'.base64_encode($content);
        } catch (\Exception $e) {
            return null;
        }
    }

    public static function formatMoney($amount, ?array $settings = null): string
    {
        $amount = is_numeric($amount) ? (float) $amount : 0.0;

        $currency = self::currencyMeta($settings);

        $precision = (int) session('business.currency_precision', 2);
        $placement = (string) session('business.currency_symbol_placement', 'before');

        $formatted = number_format(
            $amount,
            $precision,
            (string) ($currency['decimal_separator'] ?? '.'),
            (string) ($currency['thousand_separator'] ?? ',')
        );

        $symbol = trim((string) ($currency['symbol'] ?? ''));
        if ($symbol === '') {
            return $formatted;
        }

        return $placement === 'after'
            ? $formatted.' '.$symbol
            : $symbol.' '.$formatted;
    }

    public static function currencyMeta(?array $settings = null): array
    {
        $fallback = session()->has('currency')
            ? (array) session('currency')
            : [
                'symbol' => '',
                'thousand_separator' => ',',
                'decimal_separator' => '.',
            ];

        $currencyId = (int) (($settings ?? [])['currency_id'] ?? 0);
        if ($currencyId <= 0) {
            return $fallback;
        }

        $row = Currency::query()->find($currencyId);
        if (empty($row)) {
            return $fallback;
        }

        return [
            'symbol' => (string) $row->symbol,
            'thousand_separator' => (string) $row->thousand_separator,
            'decimal_separator' => (string) $row->decimal_separator,
        ];
    }
}
