<?php

namespace Modules\Mosque\Utils;

class TenancyKeyUtil
{
    public static function normalizePhone(string $phone): string
    {
        $digits = preg_replace('/\D+/', '', trim($phone));
        return (string) $digits;
    }

    /**
     * Generate multiple possible keys (phone-only, nid-only, etc.) to improve matching.
     *
     * @return array<int, string>
     */
    public static function possibleKeys(string $phone, string $nidNo = '', string $passportNo = ''): array
    {
        $phone = self::normalizePhone($phone);
        $nidNo = preg_replace('/\s+/', '', strtoupper(trim($nidNo)));
        $passportNo = preg_replace('/\s+/', '', strtoupper(trim($passportNo)));

        if ($phone === '' && $nidNo === '' && $passportNo === '') {
            return [];
        }

        $phones = [];
        if ($phone !== '') {
            $phones[] = $phone;
            if (strlen($phone) === 11 && str_starts_with($phone, '0')) {
                $phones[] = '880'.substr($phone, 1);
            }
            if (strlen($phone) === 10 && str_starts_with($phone, '1')) {
                $phones[] = '880'.$phone;
            }
        } else {
            $phones[] = '';
        }

        $phones = array_values(array_unique($phones));

        $keys = [];
        foreach ($phones as $p) {
            $candidates = [
                [$p, '', ''],
                ['', $nidNo, ''],
                ['', '', $passportNo],
                [$p, $nidNo, ''],
                [$p, '', $passportNo],
                ['', $nidNo, $passportNo],
                [$p, $nidNo, $passportNo],
            ];
            foreach ($candidates as $parts) {
                [$pp, $nn, $ss] = $parts;
                if ($pp === '' && $nn === '' && $ss === '') {
                    continue;
                }
                $keys[] = hash('sha256', $pp.'|'.$nn.'|'.$ss);
            }
        }

        return array_values(array_unique($keys));
    }

    public static function tenantKey(string $phone, string $nidNo = '', string $passportNo = ''): string
    {
        $phone = self::normalizePhone($phone);
        $nidNo = preg_replace('/\s+/', '', strtoupper(trim($nidNo)));
        $passportNo = preg_replace('/\s+/', '', strtoupper(trim($passportNo)));

        if ($phone === '' && $nidNo === '' && $passportNo === '') {
            return '';
        }

        return hash('sha256', $phone.'|'.$nidNo.'|'.$passportNo);
    }
}
