<?php

namespace Modules\Mosque\Utils;

use App\Business;
use App\Utils\Util;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Entities\TenAllocation;
use Modules\Mosque\Entities\TenBill;
use Modules\Mosque\Entities\TenTenant;

class TenancyNoticeUtil
{
    private static function settings(int $businessId): array
    {
        if (! Schema::hasTable('mosque_settings')) {
            return [];
        }
        return (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
    }

    private static function mosqueName(int $businessId): string
    {
        $businessName = (string) request()->session()->get('business.name')
            ?: (string) request()->session()->get('business.business_name')
            ?: 'Business';

        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
            if (! empty($profile?->name)) {
                return (string) $profile->name;
            }
        }

        return $businessName;
    }

    private static function tokens(int $businessId, TenTenant $tenant, array $extra = []): array
    {
        $mosque = self::mosqueName($businessId);
        $tokens = [
            '{tenant}' => (string) ($tenant->full_name ?? ''),
            '{phone}' => (string) ($tenant->phone ?? ''),
            '{mosque}' => $mosque,
            '{business}' => $mosque,
            '{date}' => now()->format('Y-m-d'),
        ];

        foreach ($extra as $k => $v) {
            $tokens['{'.$k.'}'] = (string) $v;
        }

        return $tokens;
    }

    private static function sendSms(int $businessId, string $phone, string $body): bool
    {
        $phone = trim($phone);
        if ($phone === '' || trim($body) === '') {
            return false;
        }

        try {
            $business = Business::findOrFail($businessId);
            $util = new Util();
            $util->sendSms([
                'sms_settings' => $business->sms_settings,
                'mobile_number' => $phone,
                'sms_body' => $body,
            ]);
            return true;
        } catch (\Throwable $e) {
            return false;
        }
    }

    private static function whatsappLink(int $businessId, string $phone, string $body): string
    {
        $digits = preg_replace('/\D+/', '', (string) $phone);
        if ($digits === '' || trim($body) === '') {
            return '';
        }

        try {
            $business = Business::findOrFail($businessId);
            $util = new Util();
            $data = [
                'business_id' => $businessId,
                'sms_settings' => $business->sms_settings,
                'email_settings' => $business->email_settings,
                'mobile_number' => $digits,
                'whatsapp_text' => $body,
            ];
            $url = (string) $util->getWhatsappNotificationAPI($data);
            if ($url !== '') {
                return $url;
            }
        } catch (\Throwable $e) {
        }

        return 'https://wa.me/'.$digits.'?text='.rawurlencode($body);
    }

    private static function sendEmail(string $email, string $subject, string $body): bool
    {
        $email = trim($email);
        if ($email === '' || trim($body) === '') {
            return false;
        }
        try {
            Mail::send([], [], function ($message) use ($email, $subject, $body) {
                $message->to($email)
                    ->subject($subject ?: 'Notification')
                    ->setBody($body, 'text/html');
            });
            return true;
        } catch (\Throwable $e) {
            return false;
        }
    }

    public static function billGenerated(int $businessId, TenTenant $tenant, TenBill $bill): array
    {
        $settings = self::settings($businessId);
        $tokens = self::tokens($businessId, $tenant, [
            'amount' => (string) ($bill->amount ?? ''),
            'period_from' => (string) ($bill->period_from ?? ''),
            'period_to' => (string) ($bill->period_to ?? ''),
        ]);

        $smsEnabled = (bool) ($settings['tenancy_notice_sms_enabled'] ?? true);
        $waEnabled = (bool) ($settings['tenancy_notice_whatsapp_enabled'] ?? true);
        $emailEnabled = (bool) ($settings['tenancy_notice_email_enabled'] ?? false);

        $smsTpl = (string) ($settings['tenancy_template_bill_generated_sms'] ?? '');
        if (trim($smsTpl) === '') {
            $smsTpl = 'Rent bill generated: {amount} for {period_from} to {period_to}. {mosque}';
        }
        $waTpl = (string) ($settings['tenancy_template_bill_generated_whatsapp'] ?? '');
        if (trim($waTpl) === '') {
            $waTpl = "Assalamu Alaikum {tenant},\nRent bill generated: {amount}\nPeriod: {period_from} to {period_to}\n{mosque}";
        }
        $emailSub = (string) ($settings['tenancy_template_bill_generated_email_subject'] ?? '');
        if (trim($emailSub) === '') {
            $emailSub = 'Rent bill generated';
        }
        $emailTpl = (string) ($settings['tenancy_template_bill_generated_email'] ?? '');
        if (trim($emailTpl) === '') {
            $emailTpl = 'Assalamu Alaikum {tenant},<br>Rent bill generated: <strong>{amount}</strong><br>Period: {period_from} to {period_to}<br><br>{mosque}';
        }

        $smsBody = strtr($smsTpl, $tokens);
        $waBody = strtr($waTpl, $tokens);
        $emailBody = strtr($emailTpl, $tokens);
        $emailSubject = strtr($emailSub, $tokens);

        $smsSent = $smsEnabled ? self::sendSms($businessId, (string) $tenant->phone, $smsBody) : false;
        $waUrl = $waEnabled ? self::whatsappLink($businessId, (string) ($tenant->whatsapp ?: $tenant->phone), $waBody) : '';
        $emailSent = $emailEnabled ? self::sendEmail((string) $tenant->email, $emailSubject, $emailBody) : false;

        $links = [];
        if ($waUrl !== '') {
            $links[] = ['name' => (string) ($tenant->full_name ?? 'Tenant'), 'url' => $waUrl];
        }

        return [
            'sms_sent' => $smsSent,
            'email_sent' => $emailSent,
            'whatsapp_links' => $links,
        ];
    }

    public static function allocationApproved(int $businessId, TenTenant $tenant, ?TenAllocation $allocation): array
    {
        $settings = self::settings($businessId);
        $tokens = self::tokens($businessId, $tenant, [
            'start_date' => (string) ($allocation?->start_date ?? ''),
        ]);

        $waEnabled = (bool) ($settings['tenancy_notice_whatsapp_enabled'] ?? true);
        $waTpl = (string) ($settings['tenancy_template_allocation_approved_whatsapp'] ?? '');
        if (trim($waTpl) === '') {
            $waTpl = "Assalamu Alaikum {tenant},\nYour tenancy allocation has been approved.\nStart: {start_date}\n{mosque}";
        }
        $waBody = strtr($waTpl, $tokens);
        $waUrl = $waEnabled ? self::whatsappLink($businessId, (string) ($tenant->whatsapp ?: $tenant->phone), $waBody) : '';

        $links = [];
        if ($waUrl !== '') {
            $links[] = ['name' => (string) ($tenant->full_name ?? 'Tenant'), 'url' => $waUrl];
        }

        return [
            'whatsapp_links' => $links,
        ];
    }

    public static function requestReceived(int $businessId, string $name, string $phone, string $trackingCode): array
    {
        $settings = self::settings($businessId);
        $tenant = new TenTenant([
            'full_name' => $name,
            'phone' => $phone,
            'whatsapp' => $phone,
        ]);

        $tokens = self::tokens($businessId, $tenant, [
            'tracking' => $trackingCode,
        ]);

        $smsEnabled = (bool) ($settings['tenancy_notice_sms_enabled'] ?? true);
        $waEnabled = (bool) ($settings['tenancy_notice_whatsapp_enabled'] ?? true);

        $smsTpl = (string) ($settings['tenancy_template_request_received_sms'] ?? '');
        if (trim($smsTpl) === '') {
            $smsTpl = 'Request received. Tracking: {tracking}. {mosque}';
        }

        $waTpl = (string) ($settings['tenancy_template_request_received_whatsapp'] ?? '');
        if (trim($waTpl) === '') {
            $waTpl = "Assalamu Alaikum {tenant},\nYour request was received.\nTracking: {tracking}\n{mosque}";
        }

        $smsBody = strtr($smsTpl, $tokens);
        $waBody = strtr($waTpl, $tokens);

        $smsSent = $smsEnabled ? self::sendSms($businessId, $phone, $smsBody) : false;
        $waUrl = $waEnabled ? self::whatsappLink($businessId, $phone, $waBody) : '';

        $links = [];
        if ($waUrl !== '') {
            $links[] = ['name' => $name ?: 'Requester', 'url' => $waUrl];
        }

        return [
            'sms_sent' => $smsSent,
            'whatsapp_links' => $links,
        ];
    }
}
