<?php

namespace Modules\Superadmin\Entities;

use App\User;
use App\Business;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class SupportTicket extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'ticket_number',
        'business_id',
        'created_by',
        'subject',
        'description',
        'priority',
        'category',
        'status',
        'external_reference',
        'whatsapp_ticket_number',
        'assigned_to',
        'last_response_at',
        'resolved_at',
        'resolved_by',
        'resolution_note',
    ];

    protected $casts = [
        'last_response_at' => 'datetime',
        'resolved_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Generate unique ticket number
     */
    public static function generateTicketNumber()
    {
        $prefix = 'TKT';
        $timestamp = now()->format('ymd');
        $random = str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        
        $ticket_number = "{$prefix}-{$timestamp}-{$random}";
        
        // Ensure uniqueness
        while (self::where('ticket_number', $ticket_number)->exists()) {
            $random = str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
            $ticket_number = "{$prefix}-{$timestamp}-{$random}";
        }
        
        return $ticket_number;
    }

    /**
     * Get the business that owns the ticket
     */
    public function business()
    {
        return $this->belongsTo(Business::class, 'business_id');
    }

    /**
     * Get the user who created the ticket
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the superadmin assigned to the ticket
     */
    public function assignedTo()
    {
        return $this->belongsTo(User::class, 'assigned_to');
    }

    /**
     * Get the user who resolved the ticket
     */
    public function resolver()
    {
        return $this->belongsTo(User::class, 'resolved_by');
    }

    /**
     * Get all messages for the ticket
     */
    public function messages()
    {
        return $this->hasMany(SupportTicketMessage::class, 'ticket_id')->orderBy('created_at', 'asc');
    }

    /**
     * Get all attachments for the ticket
     */
    public function attachments()
    {
        return $this->hasMany(SupportTicketAttachment::class, 'ticket_id');
    }

    /**
     * Get priority badge HTML
     */
    public function getPriorityBadgeAttribute()
    {
        $badges = [
            'low' => ['class' => 'label-default', 'icon' => 'fa-arrow-down', 'text' => 'Low'],
            'medium' => ['class' => 'label-primary', 'icon' => 'fa-minus', 'text' => 'Medium'],
            'high' => ['class' => 'label-warning', 'icon' => 'fa-arrow-up', 'text' => 'High'],
            'urgent' => ['class' => 'label-danger', 'icon' => 'fa-exclamation-triangle', 'text' => 'Urgent'],
        ];

        $badge = $badges[$this->priority] ?? ['class' => 'label-default', 'icon' => 'fa-question', 'text' => ucfirst($this->priority)];
        
        return '<span class="label ' . $badge['class'] . '"><i class="fa ' . $badge['icon'] . '"></i> ' . $badge['text'] . '</span>';
    }

    /**
     * Get status badge HTML
     */
    public function getStatusBadgeAttribute()
    {
        $badges = [
            'open' => ['class' => 'label-info', 'icon' => 'fa-folder-open', 'text' => 'Open'],
            'in_progress' => ['class' => 'label-primary', 'icon' => 'fa-spinner', 'text' => 'In Progress'],
            'waiting_customer' => ['class' => 'label-warning', 'icon' => 'fa-clock', 'text' => 'Waiting Customer'],
            'waiting_admin' => ['class' => 'label-warning', 'icon' => 'fa-clock', 'text' => 'Waiting Admin'],
            'resolved' => ['class' => 'label-success', 'icon' => 'fa-check-circle', 'text' => 'Resolved'],
            'closed' => ['class' => 'label-default', 'icon' => 'fa-times-circle', 'text' => 'Closed'],
        ];

        $status_text = ucwords(str_replace('_', ' ', $this->status));
        $badge = $badges[$this->status] ?? ['class' => 'label-default', 'icon' => 'fa-question', 'text' => $status_text];
        
        return '<span class="label ' . $badge['class'] . '"><i class="fa ' . $badge['icon'] . '"></i> ' . $badge['text'] . '</span>';
    }

    /**
     * Get unread messages count for business
     */
    public function getUnreadBusinessMessagesCount()
    {
        return $this->messages()
            ->where('sender_type', 'superadmin')
            ->whereNull('read_at')
            ->count();
    }

    /**
     * Get unread messages count for superadmin
     */
    public function getUnreadAdminMessagesCount()
    {
        return $this->messages()
            ->where('sender_type', 'business')
            ->whereNull('read_at')
            ->count();
    }

    /**
     * Check if ticket is open
     */
    public function isOpen()
    {
        return in_array($this->status, ['open', 'in_progress', 'waiting_customer', 'waiting_admin']);
    }

    /**
     * Check if ticket is closed
     */
    public function isClosed()
    {
        return in_array($this->status, ['resolved', 'closed']);
    }

    /**
     * Scope for open tickets
     */
    public function scopeOpen($query)
    {
        return $query->whereIn('status', ['open', 'in_progress', 'waiting_customer', 'waiting_admin']);
    }

    /**
     * Scope for closed tickets
     */
    public function scopeClosed($query)
    {
        return $query->whereIn('status', ['resolved', 'closed']);
    }

    /**
     * Scope for high priority tickets
     */
    public function scopeHighPriority($query)
    {
        return $query->whereIn('priority', ['high', 'urgent']);
    }
}
