<?php

namespace Modules\Superadmin\Http\Controllers;

use App\Business;
use App\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Modules\Superadmin\Notifications\BusinessEmailVerificationNotification;
use Modules\Superadmin\Notifications\BusinessWhatsAppVerificationNotification;

class BusinessVerificationController extends Controller
{
    /**
     * Show verification status page
     */
    public function index(Request $request)
    {
        $user = auth()->user();
        $business = Business::with('owner')->find($user->business_id);

        if (!$business) {
            abort(404, 'Business not found');
        }

        // Check if both are verified
        if ($business->is_verified) {
            return redirect('/home')->with('status', [
                'success' => 1,
                'msg' => __('superadmin::lang.account_verified')
            ]);
        }

        return view('superadmin::verification.index', compact('business'));
    }

    /**
     * Verify email address
     */
    public function verifyEmail(Request $request, $token)
    {
        try {
            $business = Business::where('email_verification_token', $token)->first();

            if (!$business) {
                return view('superadmin::verification.failed', [
                    'message' => __('superadmin::lang.invalid_verification_link'),
                    'type' => 'email'
                ]);
            }

            // Check if token is expired (24 hours from when email was sent)
            if ($business->email_verification_sent_at && $business->email_verification_sent_at->addHours(24)->isPast()) {
                return view('superadmin::verification.expired', [
                    'business_id' => $business->id,
                    'type' => 'email'
                ]);
            }

            // Check verification status before any updates
            $both_verified_before = $business->is_verified;
            
            // Already verified
            if ($business->email_verified_at) {
                $message = __('superadmin::lang.email_already_verified');
            } else {
                
                // Mark as verified
                $business->email_verified_at = Carbon::now();
                $business->email_verification_token = null;

                // Check if both are verified now
                if ($business->whatsapp_verified_at) {
                    $business->is_verified = true;
                    $business->is_active = true; // Activate business
                }

                $business->save();

                // Also update user
                $user = $business->owner;
                if ($user) {
                    $user->email_verified_at = Carbon::now();
                    $user->save();
                }
                
                // Send welcome notifications if both verifications are now complete
                if (!$both_verified_before && $business->is_verified) {
                    $this->sendWelcomeNotifications($business);
                }

                $message = __('superadmin::lang.email_verified_successfully');
            }

            return view('superadmin::verification.success', [
                'message' => $message,
                'business' => $business,
                'type' => 'email',
                'both_verified' => $business->is_verified
            ]);
        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            return view('superadmin::verification.failed', [
                'message' => __('messages.something_went_wrong'),
                'type' => 'email'
            ]);
        }
    }

    /**
     * Resend email verification
     */
    public function resendEmail(Request $request)
    {
        try {
            $user = auth()->user();
            $business = Business::find($user->business_id);

            if (!$business) {
                return response()->json([
                    'success' => 0,
                    'msg' => __('messages.something_went_wrong')
                ]);
            }

            if ($business->email_verified_at) {
                return response()->json([
                    'success' => 0,
                    'msg' => __('superadmin::lang.email_already_verified')
                ]);
            }

            // Generate new token
            $token = Str::random(64);
            $business->email_verification_token = $token;
            $business->email_verification_sent_at = now();
            $business->save();

            // Send verification email
            $verification_url = route('superadmin.business.verify-email', ['token' => $token]);
            $user->notify(new BusinessEmailVerificationNotification($business, $verification_url));

            return response()->json([
                'success' => 1,
                'msg' => __('superadmin::lang.verification_email_sent')
            ]);
        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            return response()->json([
                'success' => 0,
                'msg' => __('messages.something_went_wrong')
            ]);
        }
    }

    /**
     * Send WhatsApp verification code
     */
    public function sendWhatsAppCode(Request $request)
    {
        try {
            $user = auth()->user();
            $business = Business::with('locations')->find($user->business_id);

            if (!$business) {
                return response()->json([
                    'success' => 0,
                    'msg' => __('messages.something_went_wrong')
                ]);
            }

            if ($business->whatsapp_verified_at) {
                return response()->json([
                    'success' => 0,
                    'msg' => __('superadmin::lang.whatsapp_already_verified')
                ]);
            }

            // Rate limiting: max 3 attempts per 10 minutes
            $lastSent = $business->whatsapp_verification_sent_at;
            if ($lastSent && $lastSent->addMinutes(10)->isFuture()) {
                if ($business->whatsapp_verification_attempts >= 3) {
                    return response()->json([
                        'success' => 0,
                        'msg' => __('superadmin::lang.too_many_attempts')
                    ]);
                }
            } else {
                // Reset attempts after 10 minutes
                $business->whatsapp_verification_attempts = 0;
            }

            // Generate 6-digit OTP
            $verification_code = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
            
            $business->whatsapp_verification_token = $verification_code;
            $business->whatsapp_verification_sent_at = Carbon::now();
            $business->whatsapp_verification_attempts++;
            $business->save();

            // Get contact number
            $contact_number = $user->contact_number;
            if (empty($contact_number) && $business->locations->isNotEmpty()) {
                $contact_number = $business->locations->first()->mobile;
            }

            if (empty($contact_number)) {
                \Log::error('WhatsApp verification failed: No contact number', [
                    'user_id' => $user->id,
                    'business_id' => $business->id
                ]);
                return response()->json([
                    'success' => 0,
                    'msg' => __('verification.no_mobile_number')
                ]);
            }

            \Log::info('Sending WhatsApp verification code', [
                'user_id' => $user->id,
                'business_id' => $business->id,
                'contact_number' => $contact_number,
                'verification_code' => $verification_code
            ]);

            // Send verification code
            $user->notify(new BusinessWhatsAppVerificationNotification(
                $business,
                $verification_code,
                $contact_number
            ));

            \Log::info('WhatsApp verification notification sent successfully', [
                'user_id' => $user->id,
                'business_id' => $business->id
            ]);

            return response()->json([
                'success' => 1,
                'msg' => __('superadmin::lang.verification_code_sent'),
                'masked_number' => $this->maskPhoneNumber($contact_number)
            ]);
        } catch (\Exception $e) {
            \Log::error('WhatsApp verification exception', [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => 0,
                'msg' => __('messages.something_went_wrong')
            ]);
        }
    }

    /**
     * Verify WhatsApp code
     */
    public function verifyWhatsAppCode(Request $request)
    {
        $request->validate([
            'code' => 'required|string|size:6'
        ]);

        try {
            $user = auth()->user();
            $business = Business::find($user->business_id);

            if (!$business) {
                return response()->json([
                    'success' => 0,
                    'msg' => __('messages.something_went_wrong')
                ]);
            }

            if ($business->whatsapp_verified_at) {
                return response()->json([
                    'success' => 0,
                    'msg' => __('superadmin::lang.whatsapp_already_verified')
                ]);
            }

            // Check if code expired (10 minutes)
            $sentAt = $business->whatsapp_verification_sent_at;
            if (!$sentAt || $sentAt->addMinutes(10)->isPast()) {
                return response()->json([
                    'success' => 0,
                    'msg' => __('superadmin::lang.verification_code_expired')
                ]);
            }

            // Verify code
            if ($business->whatsapp_verification_token !== $request->code) {
                return response()->json([
                    'success' => 0,
                    'msg' => __('superadmin::lang.invalid_verification_code')
                ]);
            }

            // Check verification status before update
            $both_verified_before = $business->is_verified;
            
            // Mark as verified
            $business->whatsapp_verified_at = Carbon::now();
            $business->whatsapp_verification_token = null;
            $business->whatsapp_verification_sent_at = null;
            $business->whatsapp_verification_attempts = 0;

            // Check if both are verified now
            if ($business->email_verified_at) {
                $business->is_verified = true;
                $business->is_active = true; // Activate business
            }

            $business->save();

            // Also update user
            $user->whatsapp_verified_at = Carbon::now();
            $user->whatsapp_verification_token = null;
            $user->save();
            
            // Send welcome notifications if both verifications are now complete
            if (!$both_verified_before && $business->is_verified) {
                $this->sendWelcomeNotifications($business);
            }

            return response()->json([
                'success' => 1,
                'msg' => __('superadmin::lang.whatsapp_verified_successfully'),
                'both_verified' => $business->is_verified,
                'redirect_url' => $business->is_verified ? url('/home') : null
            ]);
        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            return response()->json([
                'success' => 0,
                'msg' => __('messages.something_went_wrong')
            ]);
        }
    }
    
    /**
     * Update email address during verification
     */
    public function updateEmail(Request $request)
    {
        $request->validate([
            'new_email' => 'required|email|unique:users,email'
        ]);

        try {
            $user = auth()->user();
            $business = Business::find($user->business_id);

            if (!$business) {
                return response()->json([
                    'success' => 0,
                    'msg' => __('messages.something_went_wrong')
                ]);
            }

            // Update email and reset verification status
            $old_email = $user->email;
            $user->email = $request->new_email;
            $user->email_verified_at = null;
            $user->save();

            // Reset business verification status (no email column in business table)
            $business->email_verified_at = null;
            $business->is_verified = false;
            $business->is_active = false;
            
            // Generate new verification token
            $token = Str::random(64);
            $business->email_verification_token = $token;
            $business->email_verification_sent_at = now();
            $business->save();

            // Send verification email to new address
            $verification_url = route('superadmin.business.verify-email', ['token' => $token]);
            $user->notify(new BusinessEmailVerificationNotification($business, $verification_url));

            \Log::info('Email updated during verification', [
                'user_id' => $user->id,
                'business_id' => $business->id,
                'old_email' => $old_email,
                'new_email' => $request->new_email
            ]);

            return response()->json([
                'success' => 1,
                'msg' => __('superadmin::lang.email_updated_verification_sent'),
                'new_email' => $request->new_email
            ]);
        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            return response()->json([
                'success' => 0,
                'msg' => __('messages.something_went_wrong')
            ]);
        }
    }
    
    /**
     * Update WhatsApp number during verification
     */
    public function updateWhatsApp(Request $request)
    {
        $request->validate([
            'new_whatsapp' => 'required|string|min:10|max:20'
        ]);

        try {
            $user = auth()->user();
            $business = Business::with('locations')->find($user->business_id);

            if (!$business) {
                return response()->json([
                    'success' => 0,
                    'msg' => __('messages.something_went_wrong')
                ]);
            }

            // Update contact number and reset verification status
            $old_number = $user->contact_number;
            $user->contact_number = $request->new_whatsapp;
            $user->whatsapp_verified_at = null;
            $user->save();

            // Update business location mobile
            if ($business->locations->isNotEmpty()) {
                $location = $business->locations->first();
                $location->mobile = $request->new_whatsapp;
                $location->save();
            }

            $business->whatsapp_verified_at = null;
            $business->is_verified = false;
            $business->is_active = false;
            $business->whatsapp_verification_token = null;
            $business->whatsapp_verification_sent_at = null;
            $business->whatsapp_verification_attempts = 0;
            $business->save();

            \Log::info('WhatsApp number updated during verification', [
                'user_id' => $user->id,
                'business_id' => $business->id,
                'old_number' => $old_number,
                'new_number' => $request->new_whatsapp
            ]);

            return response()->json([
                'success' => 1,
                'msg' => __('superadmin::lang.whatsapp_updated_send_code'),
                'new_number' => $this->maskPhoneNumber($request->new_whatsapp)
            ]);
        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            return response()->json([
                'success' => 0,
                'msg' => __('messages.something_went_wrong')
            ]);
        }
    }

    /**
     * Mask phone number for display
     */
    private function maskPhoneNumber($number)
    {
        if (empty($number)) {
            return '';
        }

        $length = strlen($number);
        if ($length <= 4) {
            return $number;
        }

        $visible = 2; // Show last 2 digits
        $masked = str_repeat('*', $length - $visible);
        return $masked . substr($number, -$visible);
    }
    
    /**
     * Send welcome notifications to business owner and admin
     */
    protected function sendWelcomeNotifications($business)
    {
        try {
            $user = User::find($business->owner_id);
            
            if (!$user) {
                \Log::error('User not found for welcome notifications', ['business_id' => $business->id]);
                return;
            }
            
            // Send welcome notification using existing Superadmin notification
            // This handles both email and WhatsApp via WhatsAppChannel
            $email_data = [
                'subject' => 'Welcome to ' . config('app.name') . '!',
                'body' => '<h2>🎉 Congratulations!</h2>
                          <p>Your business account <strong>' . $business->name . '</strong> has been successfully verified.</p>
                          <h3>Next Steps:</h3>
                          <ul>
                              <li>Complete your business profile</li>
                              <li>Choose a subscription package to get started</li>
                              <li>Add your products and services</li>
                              <li>Configure your business settings</li>
                          </ul>
                          <p>We\'re excited to help you grow your business!</p>
                          <p><strong>Need help?</strong> Contact our support team at any time.</p>'
            ];
            
            $user->notify(new \Modules\Superadmin\Notifications\NewBusinessWelcomNotification($email_data));
            
            // Notify admin about new business registration
            $this->notifyAdminNewBusiness($business);
            
            \Log::info('Welcome notifications sent successfully', [
                'business_id' => $business->id,
                'business_name' => $business->name
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Failed to send welcome notifications: ' . $e->getMessage(), [
                'business_id' => $business->id,
                'exception' => $e->getTraceAsString()
            ]);
        }
    }
    
    /**
     * Notify admin about new business registration
     */
    protected function notifyAdminNewBusiness($business)
    {
        try {
            // Get admin email from config or superadmin settings
            $admin_email = config('mail.from.address');
            
            // Send notification to admin using existing Superadmin notification
            if (!empty($admin_email)) {
                $admin = \App\User::where('email', $admin_email)->first();
                
                if ($admin) {
                    // Use existing Superadmin notification which handles both email and WhatsApp
                    $admin->notify(new \Modules\Superadmin\Notifications\NewBusinessNotification($business));
                }
            }
            
            \Log::info('Admin notified of new business', [
                'business_id' => $business->id,
                'admin_email' => $admin_email ?? 'N/A'
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Failed to notify admin: ' . $e->getMessage(), [
                'business_id' => $business->id
            ]);
        }
    }
}
