<?php

namespace Modules\Superadmin\Http\Controllers;

use App\Business;
use App\Charts\CommonChart;
use App\System;
use Carbon;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;
use Modules\Superadmin\Entities\Subscription;
use Illuminate\Routing\Controller;

class SuperadminController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function index()
    {
        if (! auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        $date_filters['this_yr'] = ['start' => Carbon::today()->startOfYear()->toDateString(),
            'end' => Carbon::today()->endOfYear()->toDateString(),
        ];
        $date_filters['this_month']['start'] = date('Y-m-01');
        $date_filters['this_month']['end'] = date('Y-m-t');
        $date_filters['this_week']['start'] = date('Y-m-d', strtotime('monday this week'));
        $date_filters['this_week']['end'] = date('Y-m-d', strtotime('sunday this week'));

        $currency = System::getCurrency();

        //Count all busineses not subscribed.
        $not_subscribed = Business::leftjoin('subscriptions AS s', 'business.id', '=', 's.business_id')
            ->whereNull('s.id')
            ->count();

        $subscriptions = $this->_monthly_sell_data();

        $monthly_sells_chart = new CommonChart;
        $monthly_sells_chart->labels(array_keys($subscriptions))
            ->dataset(__('superadmin::lang.total_subscriptions', ['currency' => $currency->currency]), 'column', array_values($subscriptions));

        return view('superadmin::superadmin.index')
            ->with(compact(
                'date_filters',
                'not_subscribed',
                'monthly_sells_chart'
            ));
    }

    /**
     * Returns the monthly sell data for chart
     *
     * @return array
     */
    protected function _monthly_sell_data()
    {
        $start = Carbon::today()->subYear();
        $end = Carbon::today();
        $subscriptions = Subscription::whereRaw('DATE(created_at) BETWEEN ? AND ?', [$start, $end])
            ->select('package_price', 'created_at')
            ->where('status', 'approved')
            ->orderBy('created_at')
            ->get();
        $subscription_formatted = [];
        foreach ($subscriptions as $value) {
            $month_year = Carbon::createFromFormat('Y-m-d H:i:s', $value->created_at)->format('M-Y');
            if (! isset($subscription_formatted[$month_year])) {
                $subscription_formatted[$month_year] = 0;
            }
            $subscription_formatted[$month_year] += (float) $value->package_price;
        }

        return $subscription_formatted;
    }

    /**
     * Returns the stats for superadmin
     *
     * @param $start date
     * @param $end date
     * @return json
     */
    public function stats(Request $request)
    {
        if (! auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        $start_date = $request->get('start');
        $end_date = $request->get('end');

        $subscription = Subscription::whereRaw('DATE(created_at) BETWEEN ? AND ?', [$start_date, $end_date])
            ->where('status', 'approved')
            ->select(DB::raw('SUM(package_price) as total'))
            ->first()
            ->total;

        $registrations = Business::whereRaw('DATE(created_at) BETWEEN ? AND ?', [$start_date, $end_date])
            ->select(DB::raw('COUNT(id) as total'))
            ->first()
            ->total;

        return ['new_subscriptions' => $subscription,
            'new_registrations' => $registrations,
        ];
    }

    /**
     * Get active subscription for the authenticated business
     * Enhanced API with three response modes:
     * 1. SaaS Mode: Return full subscription details with validation
     * 2. On-Premise Mode: Return package_details with all modules enabled
     * 3. Inactive Mode: Return subscription_status for inactive/expired subscriptions
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getActiveSubscription()
    {
        try {
            // Get business_id from authenticated user (API auth doesn't use sessions)
            $user = auth()->user();
            
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'subscription_status' => 'inactive',
                    'message' => 'Unauthorized'
                ], 401);
            }
            
            $business_id = $user->business_id;
            
            if (!$business_id) {
                return response()->json([
                    'success' => false,
                    'subscription_status' => 'inactive',
                    'message' => 'Business not found'
                ], 404);
            }

            // Get business first to verify it exists
            $business = Business::find($business_id);
            
            if (!$business) {
                return response()->json([
                    'success' => false,
                    'subscription_status' => [
                        'status' => 'inactive',
                        'message' => 'Business not found',
                        'action_required' => 'contact_admin'
                    ],
                    'message' => 'Business not found'
                ], 403);
            }

            // ==========================================
            // ON-PREMISE MODE DISABLED - 100% SaaS Only
            // ==========================================
            // Note: This system is configured as 100% SaaS
            // All businesses MUST have valid subscriptions
            // On-premise logic has been commented out
            // ==========================================
            
            /*
            // COMMENTED OUT: On-premise detection (system is 100% SaaS)
            $superadminEnabled = \App\System::getProperty('superadmin_enable');
            $isSaas = $superadminEnabled == '1' || $superadminEnabled == 1;

            if (!$isSaas) {
                \Log::info('[Subscription] On-premise mode detected for business_id: ' . $business_id);
                
                $enabled_modules = $business->enabled_modules ?? [];
                if (is_string($enabled_modules)) {
                    $enabled_modules = json_decode($enabled_modules, true) ?: [];
                }
                if (!is_array($enabled_modules)) {
                    $enabled_modules = [];
                }
                
                return response()->json([
                    'success' => true,
                    'package_details' => [
                        'name' => 'On-Premise',
                        'location_count' => 0,
                        'user_count' => 0,
                        'product_count' => 0,
                        'invoice_count' => 0,
                        'accounting_module' => true,
                        'crm_module' => in_array('crm', $enabled_modules),
                        'repair_module' => in_array('repair', $enabled_modules),
                        'assetmanagement_module' => in_array('assetmanagement', $enabled_modules),
                        'manufacturing_module' => in_array('manufacturing', $enabled_modules),
                        'project_module' => in_array('project', $enabled_modules),
                        'aiassistance_module' => false,
                        'connector_module' => true,
                        'essentials_module' => in_array('essentials', $enabled_modules),
                        'inventorymanagement_module' => true,
                        'productcatalogue_module' => in_array('productcatalogue', $enabled_modules),
                        'spreadsheet_module' => in_array('spreadsheet', $enabled_modules),
                        'woocommerce_module' => in_array('woocommerce', $enabled_modules),
                    ]
                ], 200);
            }
            */

            // SaaS Mode: Check for active subscription (ALWAYS REQUIRED)
            \Log::info('[Subscription] SaaS mode - checking subscription for business_id: ' . $business_id);
            
            // Get all approved/active subscriptions, ordered by most recent first
            $subscriptions = Subscription::where('business_id', $business_id)
                ->whereIn('status', ['approved', 'active'])
                ->with(['package'])
                ->orderBy('created_at', 'desc')
                ->get();

            $now = \Carbon\Carbon::now();
            $validSubscription = null;

            // Find the first non-expired subscription
            foreach ($subscriptions as $sub) {
                $endDate = $sub->end_date ?? $sub->trial_end_date;
                
                if ($endDate) {
                    $expiry = \Carbon\Carbon::parse($endDate);
                    if ($now->lessThanOrEqualTo($expiry)) {
                        // Found a valid, non-expired subscription
                        $validSubscription = $sub;
                        break;
                    }
                } else {
                    // No end date means lifetime subscription
                    $validSubscription = $sub;
                    break;
                }
            }

            // Active subscription exists
            if ($validSubscription) {
                // Return active subscription
                \Log::info('[Subscription] Active subscription found for business_id: ' . $business_id . ' (Sub ID: ' . $validSubscription->id . ')');
                return response()->json([
                    'success' => true,
                    'data' => $validSubscription
                ], 200);
            }

            // All subscriptions are expired
            if ($subscriptions->isNotEmpty()) {
                // Get the most recent expired subscription
                $latestExpired = $subscriptions->first();
                
                \Log::warning('[Subscription] All subscriptions expired for business_id: ' . $business_id);
                return response()->json([
                    'success' => false,
                    'subscription_status' => [
                        'status' => 'expired',
                        'message' => 'Subscription has expired',
                        'action_required' => 'renew_subscription',
                        'end_date' => $latestExpired->end_date,
                        'trial_end_date' => $latestExpired->trial_end_date
                    ],
                    'message' => 'Subscription has expired'
                ], 403);
            }

            // No active subscription found in SaaS mode
            \Log::warning('[Subscription] No active subscription found for business_id: ' . $business_id);
            
            // Inactive Mode: No subscription found
            return response()->json([
                'success' => false,
                'subscription_status' => [
                    'status' => 'inactive',
                    'message' => 'No active subscription found',
                    'action_required' => 'contact_admin'
                ],
                'message' => 'No active subscription found'
            ], 403);

        } catch (\Exception $e) {
            \Log::error('Get Active Subscription Error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'subscription_status' => 'error',
                'message' => 'Failed to retrieve subscription: ' . $e->getMessage()
            ], 500);
        }
    }
}
