<?php

namespace Modules\Superadmin\Http\Controllers;

use App\Utils\BusinessUtil;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;
use Modules\Superadmin\Entities\Package;
use Modules\Superadmin\Entities\Subscription;
use Modules\Superadmin\Notifications\OfflinePaymentDeclinedNotification;
use Modules\Superadmin\Notifications\SubscriptionActivatedNotification;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Routing\Controller;

class SuperadminSubscriptionsController extends BaseController
{
    protected $businessUtil;

    /**
     * Constructor
     *
     * @param  BusinessUtil  $businessUtil
     * @return void
     */
    public function __construct(BusinessUtil $businessUtil)
    {
        $this->businessUtil = $businessUtil;
    }

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function index()
    {
        if (! auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }


        if (request()->ajax()) {
            $superadmin_subscription = Subscription::join('business', 'subscriptions.business_id', '=', 'business.id')
                ->join('packages', 'subscriptions.package_id', '=', 'packages.id')
                ->select('business.name as business_name', 'packages.name as package_name', 'subscriptions.status',
                 'subscriptions.created_at', 'subscriptions.start_date', 'subscriptions.trial_end_date', 'subscriptions.end_date', 'subscriptions.coupon_code','subscriptions.original_price', 'subscriptions.package_price', 'subscriptions.paid_via', 'subscriptions.payment_transaction_id', 'subscriptions.proof_of_payment', 'subscriptions.payment_details', 'subscriptions.id');

            if(!empty(request()->input('status'))) {
                $superadmin_subscription->where('subscriptions.status', request()->input('status'));
            }
            if(!empty(request()->input('package_id'))) {
                $superadmin_subscription->where('packages.id', request()->input('package_id'));
            }

            if (!empty(request()->start_date) && !empty(request()->end_date)) {
                $start = request()->start_date;
                $end =  request()->end_date;
                $superadmin_subscription->whereDate('subscriptions.created_at', '>=', $start)
                    ->whereDate('subscriptions.created_at', '<=', $end);
            }
            
            return DataTables::of($superadmin_subscription)
                        ->addColumn(
                            'action',
                            '<button data-href ="{{action(\'\Modules\Superadmin\Http\Controllers\SuperadminSubscriptionsController@edit\',[$id])}}" class="tw-dw-btn tw-dw-btn-xs tw-dw-btn-outline  tw-dw-btn-info change_status" data-toggle="modal" data-target="#statusModal">
                            @lang( "superadmin::lang.status")
                            </button> <button data-href ="{{action(\'\Modules\Superadmin\Http\Controllers\SuperadminSubscriptionsController@editSubscription\',["id" => $id])}}" class="tw-dw-btn tw-dw-btn-xs tw-dw-btn-outline  tw-dw-btn-primary btn-modal tw-m-1" data-container=".view_modal">
                            @lang( "messages.edit")
                            </button>'
                        )
                        ->addColumn(
                            'proof_of_payment_display',
                            function ($row) {
                                if (!empty($row->proof_of_payment)) {
                                    $url = route('superadmin.subscription.view-proof', $row->id);
                                    return '<a href="' . $url . '" target="_blank" class="btn btn-xs btn-info">
                                                <i class="fa fa-file"></i> ' . __('superadmin::lang.view_proof') . '
                                            </a>';
                                }
                                return '<span class="text-muted">' . __('superadmin::lang.not_uploaded') . '</span>';
                            }
                        )
                        ->editColumn('created_at', '{{@format_datetime($created_at)}}')
                        ->editColumn('trial_end_date', '@if(!empty($trial_end_date)){{@format_date($trial_end_date)}} @endif')
                        ->editColumn('start_date', '@if(!empty($start_date)){{@format_date($start_date)}}@endif')
                        ->editColumn('end_date', '@if(!empty($end_date)){{@format_date($end_date)}}@endif')
                        ->editColumn(
                            'status',
                            '@if($status == "approved")
                                <span class="label bg-light-green">{{__(\'superadmin::lang.\'.$status)}}
                                </span>
                            @elseif($status == "waiting")
                                <span class="label bg-aqua">{{__(\'superadmin::lang.\'.$status)}}
                                </span>
                            @else($status == "declined")
                                <span class="label bg-red">{{__(\'superadmin::lang.\'.$status)}}
                                </span>
                            @endif'
                        )
                        ->editColumn(
                            'package_price',
                            '<span class="display_currency" data-currency_symbol="true">
                                {{$package_price}}
                            </span>'
                        )
                        ->editColumn(
                            'original_price',
                            '<span class="display_currency" data-currency_symbol="true">
                                {{$original_price}}
                            </span>'
                        )
                        ->editColumn(
                            'payment_details',
                            function ($row) {
                                if (!empty($row->payment_details)) {
                                    return '<small>' . nl2br(e($row->payment_details)) . '</small>';
                                }
                                return '-';
                            }
                        )
                        ->removeColumn('id')
                        ->removeColumn('proof_of_payment')
                        ->rawColumns([2, 8, 9, 12, 13, 14])
                        ->make(false);
        }

        $packages = Package::listPackages()->pluck('name', 'id');

        $subscription_statuses = [
            'approved' => __('superadmin::lang.approved'),
            'waiting' => __('superadmin::lang.waiting'),
            'declined' => __('superadmin::lang.declined'),
        ];

        return view('superadmin::superadmin_subscription.index')
                    ->with(compact('packages', 'subscription_statuses'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        $business_id = request()->input('business_id');
        $packages = Package::active()->orderby('sort_order')->pluck('name', 'id');

        $gateways = $this->_payment_gateways();

        return view('superadmin::superadmin_subscription.add_subscription')
              ->with(compact('packages', 'business_id', 'gateways'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  Request  $request
     * @return Response
     */
    public function store(Request $request)
    {
        if (! auth()->user()->can('subscribe')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $input = $request->only(['business_id', 'package_id', 'paid_via', 'payment_transaction_id']);
            $package = Package::find($input['package_id']);
            $user_id = $request->session()->get('user.id');

            $subscription = $this->_add_subscription(null,$package->price ,$input['business_id'], $package, $input['paid_via'], $input['payment_transaction_id'], $user_id, true);

            DB::commit();

            $output = ['success' => 1,
                'msg' => __('lang_v1.success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();

            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => 0, 'msg' => __('messages.something_went_wrong')];
        }

        return back()->with('status', $output);
    }

    /**
     * Show the specified resource.
     *
     * @return Response
     */
    public function show()
    {
        return view('superadmin::show');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @return Response
     */
    public function edit($id)
    {
        if (! auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $status = Subscription::package_subscription_status();
            $subscription = Subscription::find($id);

            return view('superadmin::superadmin_subscription.edit')
                        ->with(compact('subscription', 'status'));
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  Request  $request
     * @return Response
     */
    public function update(Request $request, $id)
    {
        if (! auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            try {
                $input = $request->only(['status', 'payment_transaction_id']);

                $subscriptions = Subscription::findOrFail($id);

                if ($subscriptions->status != 'approved' && empty($subscriptions->start_date) && $input['status'] == 'approved') {
                    $dates = $this->_get_package_dates($subscriptions->business_id, $subscriptions->package);
                    $subscriptions->start_date = $dates['start'];
                    $subscriptions->end_date = $dates['end'];
                    $subscriptions->trial_end_date = $dates['trial'];
                }

                $subscriptions->status = $input['status'];
                $subscriptions->payment_transaction_id = $input['payment_transaction_id'];
                $subscriptions->save();

                // Send notification to customer when subscription is activated
                if ($input['status'] == 'approved' && $subscriptions->business && $subscriptions->business->owner) {
                    try {
                        $subscriptions->business->owner->notify(new SubscriptionActivatedNotification($subscriptions));
                    } catch (\Exception $e) {
                        \Log::error('Failed to send subscription activation notification: ' . $e->getMessage());
                    }
                }

                // Send notification to customer when offline payment is declined
                if ($input['status'] == 'declined' && $subscriptions->business && $subscriptions->business->owner) {
                    try {
                        $subscriptions->business->owner->notify(new OfflinePaymentDeclinedNotification($subscriptions));
                    } catch (\Exception $e) {
                        \Log::error('Failed to send offline payment declined notification: ' . $e->getMessage());
                    }
                }

                $output = ['success' => true,
                    'msg' => __('superadmin::lang.subcription_updated_success'),
                ];
            } catch (\Exception $e) {
                \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

                $output = ['success' => false,
                    'msg' => __('messages.something_went_wrong'),
                ];
            }

            return $output;
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @return Response
     */
    public function destroy()
    {
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @return Response
     */
    public function editSubscription($id)
    {
        if (! auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $subscription = Subscription::find($id);

            return view('superadmin::superadmin_subscription.edit_date_modal')
                        ->with(compact('subscription'));
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  Request  $request
     * @return Response
     */
    public function updateSubscription(Request $request)
    {
        if (! auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            try {
                $input = $request->only(['start_date', 'end_date', 'trial_end_date']);

                $subscription = Subscription::findOrFail($request->input('subscription_id'));

                $subscription->start_date = ! empty($input['start_date']) ? $this->businessUtil->uf_date($input['start_date']) : null;
                $subscription->end_date = ! empty($input['end_date']) ? $this->businessUtil->uf_date($input['end_date']) : null;
                $subscription->trial_end_date = ! empty($input['trial_end_date']) ? $this->businessUtil->uf_date($input['trial_end_date']) : null;
                $subscription->save();

                $output = ['success' => true,
                    'msg' => __('superadmin::lang.subcription_updated_success'),
                ];
            } catch (\Exception $e) {
                \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

                $output = ['success' => false,
                    'msg' => __('messages.something_went_wrong'),
                ];
            }

            return $output;
        }
    }

    /**
     * View proof of payment file
     *
     * @param  int  $id
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function viewProofOfPayment($id)
    {
        try {
            $subscription = Subscription::findOrFail($id);
            
            if (empty($subscription->proof_of_payment)) {
                abort(404, 'Proof of payment not found');
            }

            // Get the full path to the file
            $filePath = storage_path('app/public/' . $subscription->proof_of_payment);
            
            // Check if file exists
            if (!file_exists($filePath)) {
                abort(404, 'Proof of payment file not found on server');
            }

            // Get file extension and set appropriate content type
            $extension = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));
            $contentType = match($extension) {
                'pdf' => 'application/pdf',
                'jpg', 'jpeg' => 'image/jpeg',
                'png' => 'image/png',
                default => 'application/octet-stream',
            };

            // Return file response
            return response()->file($filePath, [
                'Content-Type' => $contentType,
                'Content-Disposition' => 'inline; filename="' . basename($filePath) . '"',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error viewing proof of payment: ' . $e->getMessage());
            abort(404, 'Unable to load proof of payment');
        }
    }
}
