<?php

namespace Modules\Superadmin\Http\Controllers;

use App\Utils\ModuleUtil;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Modules\Superadmin\Entities\SupportTicket;
use Modules\Superadmin\Entities\SupportTicketMessage;
use Modules\Superadmin\Entities\SupportTicketAttachment;
use Modules\Superadmin\Services\WhatsAppTicketService;
use App\User;

class SuperadminSupportTicketController extends Controller
{
    protected $whatsappService;
    protected $moduleUtil;

    public function __construct(WhatsAppTicketService $whatsappService, ModuleUtil $moduleUtil)
    {
        $this->middleware(['web', 'auth', 'SetSessionData', 'language', 'timezone']);
        $this->whatsappService = $whatsappService;
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display the ticket dashboard
     */
    public function index(Request $request)
    {
        if (!auth()->user()->can('superadmin.manage_tickets')) {
            abort(403, 'Unauthorized action.');
        }

        $status_filter = $request->get('status', 'all');
        $priority_filter = $request->get('priority', 'all');
        $category_filter = $request->get('category', 'all');
        $assigned_filter = $request->get('assigned', 'all');
        $search = $request->get('search');

        $query = SupportTicket::with(['business', 'creator', 'assignedTo', 'messages'])
            ->orderBy('created_at', 'desc');

        // Apply filters
        if ($status_filter !== 'all') {
            if ($status_filter === 'open') {
                $query->open();
            } elseif ($status_filter === 'closed') {
                $query->closed();
            } else {
                $query->where('status', $status_filter);
            }
        }

        if ($priority_filter !== 'all') {
            $query->where('priority', $priority_filter);
        }

        if ($category_filter !== 'all') {
            $query->where('category', $category_filter);
        }

        if ($assigned_filter !== 'all') {
            if ($assigned_filter === 'unassigned') {
                $query->whereNull('assigned_to');
            } elseif ($assigned_filter === 'me') {
                $query->where('assigned_to', Auth::id());
            } else {
                $query->where('assigned_to', $assigned_filter);
            }
        }

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('ticket_number', 'like', "%{$search}%")
                    ->orWhere('subject', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
            });
        }

        $tickets = $query->paginate(20);

        // Get statistics
        $stats = [
            'total' => SupportTicket::count(),
            'open' => SupportTicket::open()->count(),
            'in_progress' => SupportTicket::where('status', 'in_progress')->count(),
            'waiting_customer' => SupportTicket::where('status', 'waiting_customer')->count(),
            'resolved' => SupportTicket::where('status', 'resolved')->count(),
            'high_priority' => SupportTicket::highPriority()->count(),
            'unassigned' => SupportTicket::whereNull('assigned_to')->count(),
        ];

        // Get all admin users for assignment
        $admins = User::whereHas('roles', function ($q) {
            $q->whereIn('name', ['superadmin#1', 'Admin#1']);
        })->get();

        return view('superadmin::support_tickets.admin.index', compact(
            'tickets', 
            'stats', 
            'admins',
            'status_filter', 
            'priority_filter',
            'category_filter',
            'assigned_filter',
            'search'
        ));
    }

    /**
     * Display a specific ticket
     */
    public function show($id)
    {
        if (!auth()->user()->can('superadmin.manage_tickets')) {
            abort(403, 'Unauthorized action.');
        }

        $ticket = SupportTicket::with([
            'business', 
            'creator', 
            'assignedTo',
            'resolver',
            'messages.user', 
            'messages.attachments', 
            'attachments'
        ])->findOrFail($id);

        // Mark business messages as read
        $ticket->messages()
            ->where('sender_type', 'business')
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        // Get all admin users for assignment
        $admins = User::whereHas('roles', function ($q) {
            $q->whereIn('name', ['superadmin#1', 'Admin#1']);
        })->get();

        return view('superadmin::support_tickets.admin.show', compact('ticket', 'admins'));
    }

    /**
     * Show form for creating a new ticket on behalf of a business
     */
    public function create()
    {
        if (!auth()->user()->can('superadmin.manage_tickets')) {
            abort(403, 'Unauthorized action.');
        }

        // Get all businesses
        $businesses = \App\Business::orderBy('name')->get();

        $categories = [
            'billing' => __('superadmin::lang.billing'),
            'technical_support' => __('superadmin::lang.technical_support'),
            'feature_request' => __('superadmin::lang.feature_request'),
            'bug_report' => __('superadmin::lang.bug_report'),
            'account_issue' => __('superadmin::lang.account_issue'),
            'subscription' => __('superadmin::lang.subscription_issue'),
            'other' => __('superadmin::lang.other'),
        ];

        $priorities = [
            'low' => __('superadmin::lang.low'),
            'medium' => __('superadmin::lang.medium'),
            'high' => __('superadmin::lang.high'),
            'urgent' => __('superadmin::lang.urgent'),
        ];

        return view('superadmin::support_tickets.admin.create', compact('businesses', 'categories', 'priorities'));
    }

    /**
     * Store a new ticket created by superadmin for a business
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('superadmin.manage_tickets')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'business_id' => 'required|exists:business,id',
            'subject' => 'required|string|max:255',
            'category' => 'required|string',
            'priority' => 'required|in:low,medium,high,urgent',
            'description' => 'required|string',
            'attachments.*' => 'nullable|file|max:10240',
        ]);

        DB::beginTransaction();
        try {
            // Generate unique ticket number
            $ticketNumber = 'TKT-' . strtoupper(uniqid());

            // Create ticket
            $ticket = SupportTicket::create([
                'business_id' => $request->business_id,
                'ticket_number' => $ticketNumber,
                'subject' => $request->subject,
                'category' => $request->category,
                'priority' => $request->priority,
                'description' => $request->description,
                'status' => 'open',
                'created_by' => Auth::id(),
                'assigned_to' => Auth::id(), // Auto-assign to creator
            ]);

            // Handle file attachments
            if ($request->hasFile('attachments')) {
                foreach ($request->file('attachments') as $file) {
                    $this->storeAttachment($ticket, $file, Auth::id());
                }
            }

            // Create initial system message
            SupportTicketMessage::create([
                'ticket_id' => $ticket->id,
                'user_id' => Auth::id(),
                'message' => "Ticket created by superadmin on behalf of business: {$ticket->business->name}",
                'sender_type' => 'system',
                'is_system_message' => true,
            ]);

            // Notify business via WhatsApp if enabled
            if ($this->whatsappService->isEnabled()) {
                $business = \App\Business::find($request->business_id);
                if ($business && $business->owner && $business->owner->contact_number) {
                    try {
                        $message = "Hi {$business->owner->user_full_name},\n\n";
                        $message .= "A support ticket has been created for {$business->name}:\n\n";
                        $message .= "Ticket: {$ticketNumber}\n";
                        $message .= "Subject: {$request->subject}\n";
                        $message .= "Category: {$request->category}\n";
                        $message .= "Priority: {$request->priority}\n\n";
                        $message .= "Description: {$request->description}\n\n";
                        $message .= "You can view and respond to this ticket in your dashboard.";
                        
                        $this->whatsappService->sendMessage($business->owner->contact_number, $message);
                    } catch (\Exception $e) {
                        \Log::warning('Failed to send WhatsApp notification for ticket: ' . $e->getMessage());
                    }
                }
            }

            DB::commit();

            return redirect()->route('superadmin.support-tickets.show', $ticket->id)
                ->with('status', ['success' => true, 'msg' => __('superadmin::lang.ticket_created_successfully')]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Ticket creation failed: ' . $e->getMessage());
            
            return redirect()->back()
                ->withInput()
                ->with('status', ['success' => false, 'msg' => __('superadmin::lang.something_went_wrong')]);
        }
    }

    /**
     * Assign ticket to an admin
     */
    public function assign(Request $request, $id)
    {
        if (!auth()->user()->can('superadmin.manage_tickets')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'assigned_to' => 'required|exists:users,id',
        ]);

        $ticket = SupportTicket::findOrFail($id);
        $oldAssignee = $ticket->assigned_to;

        $ticket->update([
            'assigned_to' => $request->assigned_to,
            'status' => $ticket->status === 'open' ? 'in_progress' : $ticket->status,
        ]);

        // Create system message
        $assignee = User::find($request->assigned_to);
        $message = $oldAssignee 
            ? "Ticket reassigned from {$oldAssignee} to {$assignee->username}"
            : "Ticket assigned to {$assignee->username}";

        SupportTicketMessage::create([
            'ticket_id' => $ticket->id,
            'user_id' => Auth::id(),
            'message' => $message,
            'sender_type' => 'system',
            'is_system_message' => true,
        ]);

        return redirect()->route('superadmin.support-tickets.show', $id)
            ->with('status', ['success' => true, 'msg' => __('superadmin::lang.ticket_assigned_successfully')]);
    }

    /**
     * Add a response to the ticket
     */
    public function respond(Request $request, $id)
    {
        if (!auth()->user()->can('superadmin.manage_tickets')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'message' => 'required|string',
            'is_internal_note' => 'boolean',
            'attachments.*' => 'nullable|file|max:10240',
        ]);

        $ticket = SupportTicket::findOrFail($id);

        DB::beginTransaction();
        try {
            // Create message
            $message = SupportTicketMessage::create([
                'ticket_id' => $ticket->id,
                'user_id' => Auth::id(),
                'message' => $request->message,
                'sender_type' => 'superadmin',
                'is_internal_note' => $request->boolean('is_internal_note'),
            ]);

            // Handle attachments
            if ($request->hasFile('attachments')) {
                foreach ($request->file('attachments') as $file) {
                    $this->storeAttachment($ticket, $file, Auth::id(), $message->id);
                }
            }

            // Update ticket status and last response time
            if (!$request->boolean('is_internal_note')) {
                $ticket->update([
                    'status' => 'waiting_customer',
                    'last_response_at' => now(),
                ]);

                // Send message via WhatsApp if enabled and not an internal note
                if ($ticket->whatsapp_ticket_number && $this->whatsappService->isEnabled()) {
                    $this->whatsappService->sendMessage($ticket->whatsapp_ticket_number, $request->message);
                }
            }

            DB::commit();

            return redirect()->route('superadmin.support-tickets.show', $id)
                ->with('status', ['success' => true, 'msg' => __('superadmin::lang.response_sent_successfully')]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Ticket response failed: ' . $e->getMessage());
            
            return redirect()->back()
                ->with('status', ['success' => false, 'msg' => __('superadmin::lang.something_went_wrong')]);
        }
    }

    /**
     * Update ticket status
     */
    public function updateStatus(Request $request, $id)
    {
        if (!auth()->user()->can('superadmin.manage_tickets')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'status' => 'required|in:open,in_progress,waiting_customer,waiting_admin,resolved,closed',
        ]);

        $ticket = SupportTicket::findOrFail($id);
        $oldStatus = $ticket->status;

        $ticket->update([
            'status' => $request->status,
        ]);

        // Create system message
        SupportTicketMessage::create([
            'ticket_id' => $ticket->id,
            'user_id' => Auth::id(),
            'message' => "Status changed from {$oldStatus} to {$request->status}",
            'sender_type' => 'system',
            'is_system_message' => true,
        ]);

        // Update WhatsApp ticket if enabled
        if ($ticket->whatsapp_ticket_number && $this->whatsappService->isEnabled()) {
            $this->whatsappService->updateTicketStatus($ticket->whatsapp_ticket_number, $request->status);
        }

        return redirect()->route('superadmin.support-tickets.show', $id)
            ->with('status', ['success' => true, 'msg' => __('superadmin::lang.status_updated_successfully')]);
    }

    /**
     * Resolve the ticket
     */
    public function resolve(Request $request, $id)
    {
        if (!auth()->user()->can('superadmin.manage_tickets')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'resolution_note' => 'required|string',
        ]);

        $ticket = SupportTicket::findOrFail($id);

        DB::beginTransaction();
        try {
            $ticket->update([
                'status' => 'resolved',
                'resolved_at' => now(),
                'resolved_by' => Auth::id(),
                'resolution_note' => $request->resolution_note,
            ]);

            // Create system message
            SupportTicketMessage::create([
                'ticket_id' => $ticket->id,
                'user_id' => Auth::id(),
                'message' => "Ticket resolved: {$request->resolution_note}",
                'sender_type' => 'system',
                'is_system_message' => true,
            ]);

            // Update WhatsApp ticket if enabled
            if ($ticket->whatsapp_ticket_number && $this->whatsappService->isEnabled()) {
                $this->whatsappService->updateTicketStatus($ticket->whatsapp_ticket_number, 'resolved');
            }

            DB::commit();

            return redirect()->route('superadmin.support-tickets.show', $id)
                ->with('status', ['success' => true, 'msg' => __('superadmin::lang.ticket_resolved_successfully')]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Ticket resolution failed: ' . $e->getMessage());
            
            return redirect()->back()
                ->with('status', ['success' => false, 'msg' => __('superadmin::lang.something_went_wrong')]);
        }
    }

    /**
     * Store attachment file
     */
    protected function storeAttachment($ticket, $file, $userId, $messageId = null)
    {
        $filename = time() . '_' . $file->getClientOriginalName();
        $path = $file->storeAs('support_tickets/' . $ticket->id, $filename, 'public');

        // Determine file type
        $mimeType = $file->getMimeType();
        $fileType = 'document';
        
        if (str_starts_with($mimeType, 'image/')) {
            $fileType = 'image';
        } elseif ($mimeType === 'application/pdf') {
            $fileType = 'pdf';
        } elseif (in_array($mimeType, ['application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'])) {
            $fileType = 'spreadsheet';
        } elseif (in_array($mimeType, ['application/zip', 'application/x-rar-compressed'])) {
            $fileType = 'archive';
        }

        return SupportTicketAttachment::create([
            'ticket_id' => $ticket->id,
            'message_id' => $messageId,
            'file_name' => $file->getClientOriginalName(),
            'file_path' => $path,
            'file_type' => $fileType,
            'mime_type' => $mimeType,
            'file_size' => $file->getSize(),
            'uploaded_by' => $userId,
        ]);
    }
}
