<?php

namespace Modules\Superadmin\Http\Controllers;

use App\Utils\ModuleUtil;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Modules\Superadmin\Entities\SupportTicket;
use Modules\Superadmin\Entities\SupportTicketMessage;
use Modules\Superadmin\Entities\SupportTicketAttachment;
use Modules\Superadmin\Services\WhatsAppTicketService;

class SupportTicketController extends Controller
{
    protected $whatsappService;
    protected $moduleUtil;

    public function __construct(WhatsAppTicketService $whatsappService, ModuleUtil $moduleUtil)
    {
        $this->middleware(['web', 'auth', 'SetSessionData', 'language', 'timezone']);
        $this->whatsappService = $whatsappService;
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display a listing of tickets for the business
     */
    public function index(Request $request)
    {
        if (!auth()->user()->can('support.view_tickets')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = session('user.business_id');
        $status_filter = $request->get('status', 'all');
        $priority_filter = $request->get('priority', 'all');

        // Build query based on permissions
        $query = SupportTicket::query();
        
        // If user doesn't have view_all_business_tickets, only show tickets they created
        if (!auth()->user()->can('support.view_all_business_tickets')) {
            $query->where('business_id', $business_id)
                  ->where('created_by', Auth::id());
        } else {
            // Can view all tickets for their business
            $query->where('business_id', $business_id);
        }
        
        $query->with(['creator', 'messages'])
              ->orderBy('created_at', 'desc');

        if ($status_filter !== 'all') {
            if ($status_filter === 'open') {
                $query->open();
            } elseif ($status_filter === 'closed') {
                $query->closed();
            } else {
                $query->where('status', $status_filter);
            }
        }

        if ($priority_filter !== 'all') {
            $query->where('priority', $priority_filter);
        }

        $tickets = $query->paginate(20);

        // Get counts based on permissions
        if (!auth()->user()->can('support.view_all_business_tickets')) {
            $open_count = SupportTicket::where('business_id', $business_id)
                                       ->where('created_by', Auth::id())
                                       ->open()->count();
            $closed_count = SupportTicket::where('business_id', $business_id)
                                         ->where('created_by', Auth::id())
                                         ->closed()->count();
        } else {
            $open_count = SupportTicket::where('business_id', $business_id)->open()->count();
            $closed_count = SupportTicket::where('business_id', $business_id)->closed()->count();
        }

        return view('superadmin::support_tickets.index', compact('tickets', 'open_count', 'closed_count', 'status_filter', 'priority_filter'));
    }

    /**
     * Show the form for creating a new ticket
     */
    public function create()
    {
        if (!auth()->user()->can('support.create_ticket')) {
            abort(403, 'Unauthorized action.');
        }

        $categories = [
            'billing' => __('superadmin::lang.billing'),
            'technical_support' => __('superadmin::lang.technical_support'),
            'feature_request' => __('superadmin::lang.feature_request'),
            'bug_report' => __('superadmin::lang.bug_report'),
            'account_issue' => __('superadmin::lang.account_issue'),
            'payment_issue' => __('superadmin::lang.payment_issue'),
            'general_inquiry' => __('superadmin::lang.general_inquiry'),
            'other' => __('superadmin::lang.other'),
        ];

        $priorities = [
            'low' => __('superadmin::lang.low'),
            'medium' => __('superadmin::lang.medium'),
            'high' => __('superadmin::lang.high'),
            'urgent' => __('superadmin::lang.urgent'),
        ];

        return view('superadmin::support_tickets.create', compact('categories', 'priorities'));
    }

    /**
     * Store a newly created ticket
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('support.create_ticket')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'subject' => 'required|string|max:255',
            'description' => 'required|string',
            'category' => 'required|in:billing,technical_support,feature_request,bug_report,account_issue,payment_issue,general_inquiry,other',
            'priority' => 'required|in:low,medium,high,urgent',
            'attachments.*' => 'nullable|file|max:10240', // 10MB max per file
        ]);

        DB::beginTransaction();
        try {
            $business_id = session('user.business_id');
            $business = \App\Business::find($business_id);

            // Create ticket
            $ticket = SupportTicket::create([
                'ticket_number' => SupportTicket::generateTicketNumber(),
                'business_id' => $business_id,
                'created_by' => Auth::id(),
                'subject' => $request->subject,
                'description' => $request->description,
                'category' => $request->category,
                'priority' => $request->priority,
                'external_reference' => $request->external_reference,
                'status' => 'open',
            ]);

            // Create initial message
            SupportTicketMessage::create([
                'ticket_id' => $ticket->id,
                'user_id' => Auth::id(),
                'message' => $request->description,
                'sender_type' => 'business',
            ]);

            // Handle file attachments
            if ($request->hasFile('attachments')) {
                foreach ($request->file('attachments') as $file) {
                    $this->storeAttachment($ticket, $file, Auth::id());
                }
            }

            // Try to create WhatsApp ticket if enabled
            if ($this->whatsappService->isEnabled()) {
                $user = Auth::user();
                $phone = $business->owner->contact_number ?? $business->owner->email;
                
                $whatsappResult = $this->whatsappService->createTicket([
                    'customer_phone' => $phone,
                    'customer_name' => $business->name,
                    'subject' => $request->subject,
                    'description' => $request->description,
                    'priority' => $request->priority,
                    'category' => $request->category,
                    'external_reference' => "TICKET-{$ticket->ticket_number}",
                ]);

                if ($whatsappResult['success']) {
                    $ticket->update([
                        'whatsapp_ticket_number' => $whatsappResult['ticket_number']
                    ]);
                }
            }

            DB::commit();

            return redirect()->route('support-tickets.show', $ticket->id)
                ->with('status', ['success' => true, 'msg' => __('superadmin::lang.ticket_created_successfully')]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Ticket creation failed: ' . $e->getMessage());
            
            return redirect()->back()
                ->withInput()
                ->with('status', ['success' => false, 'msg' => __('superadmin::lang.something_went_wrong')]);
        }
    }

    /**
     * Display the specified ticket
     */
    public function show($id)
    {
        if (!auth()->user()->can('support.view_tickets')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = session('user.business_id');
        
        // Build query based on permissions
        $query = SupportTicket::query();
        
        if (!auth()->user()->can('support.view_all_business_tickets')) {
            // Can only view tickets they created
            $query->where('business_id', $business_id)
                  ->where('created_by', Auth::id());
        } else {
            // Can view all business tickets
            $query->where('business_id', $business_id);
        }
        
        $ticket = $query->with(['creator', 'messages.user', 'messages.attachments', 'attachments'])
                        ->findOrFail($id);

        // Mark superadmin messages as read
        $ticket->messages()
            ->where('sender_type', 'superadmin')
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        return view('superadmin::support_tickets.show', compact('ticket'));
    }

    /**
     * Add a reply to the ticket
     */
    public function reply(Request $request, $id)
    {
        if (!auth()->user()->can('support.reply_ticket')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'message' => 'required|string',
            'attachments.*' => 'nullable|file|max:10240',
        ]);

        $business_id = session('user.business_id');
        
        // Check if user can view all business tickets or only their own
        $query = SupportTicket::query();
        if (!auth()->user()->can('support.view_all_business_tickets')) {
            $query->where('business_id', $business_id);
        } else {
            $query->where('business_id', $business_id);
        }
        
        $ticket = $query->findOrFail($id);

        DB::beginTransaction();
        try {
            // Create message
            $message = SupportTicketMessage::create([
                'ticket_id' => $ticket->id,
                'user_id' => Auth::id(),
                'message' => $request->message,
                'sender_type' => 'business',
            ]);

            // Handle attachments
            if ($request->hasFile('attachments')) {
                foreach ($request->file('attachments') as $file) {
                    $this->storeAttachment($ticket, $file, Auth::id(), $message->id);
                }
            }

            // Update ticket status and last response time
            $ticket->update([
                'status' => 'waiting_admin',
                'last_response_at' => now(),
            ]);

            // Send message via WhatsApp if ticket has whatsapp_ticket_number
            if ($ticket->whatsapp_ticket_number && $this->whatsappService->isEnabled()) {
                $this->whatsappService->sendMessage($ticket->whatsapp_ticket_number, $request->message);
            }

            DB::commit();

            return redirect()->route('support-tickets.show', $id)
                ->with('status', ['success' => true, 'msg' => __('superadmin::lang.reply_sent_successfully')]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Ticket reply failed: ' . $e->getMessage());
            
            return redirect()->back()
                ->with('status', ['success' => false, 'msg' => __('superadmin::lang.something_went_wrong')]);
        }
    }

    /**
     * Close the ticket
     */
    public function close($id)
    {
        if (!auth()->user()->can('support.close_ticket')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = session('user.business_id');
        
        // Check if user can view all business tickets or only their own
        $query = SupportTicket::query();
        if (!auth()->user()->can('support.view_all_business_tickets')) {
            $query->where('business_id', $business_id);
        } else {
            $query->where('business_id', $business_id);
        }
        
        $ticket = $query->findOrFail($id);

        $ticket->update([
            'status' => 'closed',
        ]);

        return redirect()->route('support-tickets.show', $id)
            ->with('status', ['success' => true, 'msg' => __('superadmin::lang.ticket_closed_successfully')]);
    }

    /**
     * Store attachment file
     */
    protected function storeAttachment($ticket, $file, $userId, $messageId = null)
    {
        $filename = time() . '_' . $file->getClientOriginalName();
        $path = $file->storeAs('support_tickets/' . $ticket->id, $filename, 'public');

        // Determine file type
        $mimeType = $file->getMimeType();
        $fileType = 'document';
        
        if (str_starts_with($mimeType, 'image/')) {
            $fileType = 'image';
        } elseif ($mimeType === 'application/pdf') {
            $fileType = 'pdf';
        } elseif (in_array($mimeType, ['application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'])) {
            $fileType = 'spreadsheet';
        } elseif (in_array($mimeType, ['application/zip', 'application/x-rar-compressed'])) {
            $fileType = 'archive';
        }

        return SupportTicketAttachment::create([
            'ticket_id' => $ticket->id,
            'message_id' => $messageId,
            'file_name' => $file->getClientOriginalName(),
            'file_path' => $path,
            'file_type' => $fileType,
            'mime_type' => $mimeType,
            'file_size' => $file->getSize(),
            'uploaded_by' => $userId,
        ]);
    }
}
