<?php

namespace Modules\Superadmin\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use App\Services\WhatsAppService;
use Illuminate\Support\Facades\Artisan;

class WhatsAppSettingsController extends Controller
{
    protected $whatsappService;

    public function __construct(WhatsAppService $whatsappService)
    {
        $this->whatsappService = $whatsappService;
    }

    /**
     * Display WhatsApp settings page
     */
    public function index()
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        $settings = [
            'enabled' => config('services.whatsapp.enabled', true),
            'api_url' => config('services.whatsapp.api_url', ''),
            'token' => config('services.whatsapp.token', 'UzgsBoKOeg6Y9w1JeIda'),
        ];

        return view('superadmin::whatsapp.settings', compact('settings'));
    }

    /**
     * Update WhatsApp settings
     */
    public function update(Request $request)
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        $validated = $request->validate([
            'enabled' => 'nullable|boolean',
            'api_url' => 'required_if:enabled,1|nullable|url',
            'token' => 'required_if:enabled,1|nullable|string|min:10',
        ]);

        try {
            // Update .env file
            $this->updateEnvFile([
                'WHATSAPP_ENABLED' => $request->has('enabled') ? 'true' : 'false',
                'WHATSAPP_API_URL' => $validated['api_url'] ?? '',
                'WHATSAPP_API_TOKEN' => $validated['token'] ?? '',
            ]);

            // Clear config cache
            Artisan::call('config:clear');

            $output = [
                'success' => 1,
                'msg' => __('superadmin::lang.success')
            ];

            return redirect()->route('superadmin.whatsapp.settings')
                ->with('status', $output);

        } catch (\Exception $e) {
            \Log::error('Failed to update WhatsApp settings: ' . $e->getMessage());
            
            $output = [
                'success' => 0,
                'msg' => __('lang_v1.something_went_wrong')
            ];
            
            return back()
                ->withInput()
                ->with('status', $output);
        }
    }

    /**
     * Test WhatsApp connection
     */
    public function test(Request $request)
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        $validated = $request->validate([
            'test_phone' => 'required|string|min:10',
        ]);

        try {
            $result = $this->whatsappService->sendTestMessage($validated['test_phone']);

            if ($result['success']) {
                $output = [
                    'success' => 1,
                    'msg' => __('superadmin::lang.test_message_sent_successfully')
                ];
            } else {
                $output = [
                    'success' => 0,
                    'msg' => __('superadmin::lang.failed_to_send_test_message')
                ];
            }

            return back()->with('status', $output);

        } catch (\Exception $e) {
            \Log::error('WhatsApp test failed: ' . $e->getMessage());
            
            $output = [
                'success' => 0,
                'msg' => __('superadmin::lang.test_failed') . ': ' . $e->getMessage()
            ];
            
            return back()->with('status', $output);
        }
    }

    /**
     * Update .env file with new values
     *
     * @param array $data
     * @return void
     */
    protected function updateEnvFile(array $data)
    {
        $envPath = base_path('.env');

        if (!file_exists($envPath)) {
            throw new \Exception('.env file not found');
        }

        $envContent = file_get_contents($envPath);

        foreach ($data as $key => $value) {
            // Escape special characters in value
            $value = str_replace('"', '\"', $value);
            
            // Check if key exists in .env
            if (preg_match("/^{$key}=.*/m", $envContent)) {
                // Update existing key
                $envContent = preg_replace(
                    "/^{$key}=.*/m",
                    "{$key}=\"{$value}\"",
                    $envContent
                );
            } else {
                // Add new key
                $envContent .= "\n{$key}=\"{$value}\"";
            }
        }

        file_put_contents($envPath, $envContent);
    }
}
