<?php

namespace Modules\Superadmin\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Log;
use Modules\Superadmin\Services\WhatsAppTicketService;

class WhatsAppWebhookController extends Controller
{
    protected $whatsappService;

    public function __construct(WhatsAppTicketService $whatsappService)
    {
        $this->whatsappService = $whatsappService;
    }

    /**
     * Handle incoming WhatsApp webhook
     * This endpoint receives messages from the NearBuyMarts WhatsApp API
     * and auto-creates support tickets when keywords are detected
     */
    public function handleWebhook(Request $request)
    {
        try {
            // Verify webhook signature/token if configured
            $webhookSecret = env('WHATSAPP_WEBHOOK_SECRET');
            if ($webhookSecret) {
                $providedToken = $request->header('X-Webhook-Token') ?? $request->input('token');
                
                if ($providedToken !== $webhookSecret) {
                    Log::warning('WhatsApp webhook: Invalid token', [
                        'ip' => $request->ip(),
                    ]);
                    
                    return response()->json([
                        'success' => false,
                        'error' => 'Unauthorized',
                    ], 401);
                }
            }

            // Log incoming webhook
            Log::info('WhatsApp webhook received', [
                'payload' => $request->all(),
            ]);

            // Process the message
            $result = $this->whatsappService->processIncomingMessage($request->all());

            if ($result['success']) {
                Log::info('WhatsApp webhook processed successfully', [
                    'ticket_id' => $result['ticket']->id ?? null,
                    'ticket_number' => $result['ticket']->ticket_number ?? null,
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Ticket created successfully',
                    'ticket_number' => $result['ticket']->ticket_number ?? null,
                ], 200);
            }

            // Not a ticket request or business not found
            return response()->json([
                'success' => true,
                'message' => $result['message'] ?? 'Message processed',
                'notification_sent' => $result['notification_sent'] ?? false,
            ], 200);

        } catch (\Exception $e) {
            Log::error('WhatsApp webhook processing failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'payload' => $request->all(),
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Internal server error',
            ], 500);
        }
    }

    /**
     * Webhook verification endpoint (for initial setup)
     * Some webhook providers require verification
     */
    public function verifyWebhook(Request $request)
    {
        $challenge = $request->input('challenge');
        $verifyToken = $request->input('verify_token');
        
        $expectedToken = env('WHATSAPP_WEBHOOK_VERIFY_TOKEN', 'jebbms_whatsapp_verify');

        if ($verifyToken === $expectedToken) {
            return response($challenge, 200);
        }

        return response()->json([
            'success' => false,
            'error' => 'Invalid verification token',
        ], 403);
    }

    /**
     * Test endpoint to simulate WhatsApp message (for development/testing)
     */
    public function testMessage(Request $request)
    {
        if (config('app.env') !== 'local' && config('app.env') !== 'development') {
            abort(404);
        }

        $testData = [
            'from' => $request->input('from', '2348012345678'),
            'message' => $request->input('message', 'I need help with billing issue'),
            'sender_name' => $request->input('sender_name', 'Test User'),
        ];

        $result = $this->whatsappService->processIncomingMessage($testData);

        return response()->json($result);
    }
}
