<?php

namespace Modules\Superadmin\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Modules\Superadmin\Notifications\Channels\WhatsAppChannel;

class BusinessWhatsAppVerificationNotification extends Notification
{
    use Queueable;

    protected $business;
    protected $verification_code;
    protected $contact_number;

    /**
     * Create a new notification instance.
     *
     * @return void
     */
    public function __construct($business, $verification_code, $contact_number)
    {
        $this->business = $business;
        $this->verification_code = $verification_code;
        $this->contact_number = $contact_number;
    }

    /**
     * Get the notification's delivery channels.
     * Uses NearBuyMarts WhatsApp API via WhatsAppChannel
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        $channels = [];
        
        // Primary: Send via WhatsApp using NearBuyMarts API
        // The WhatsAppChannel uses App\Services\WhatsAppService which implements NearBuyMarts API
        if (config('services.whatsapp.enabled', false) && 
            config('services.whatsapp.api_url') && 
            config('services.whatsapp.token')) {
            $channels[] = WhatsAppChannel::class;
            
            // Only send email as fallback if WhatsApp is NOT enabled
            // This prevents duplicate messages
        // } else {
            // Fallback: Send via email only if WhatsApp is disabled
            $channels[] = 'mail';
        }
        
        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        $business_name = $this->business->name;
        $app_name = config('app.name', 'POS System');

        return (new MailMessage)
            ->subject('📱 WhatsApp Verification Code - ' . $business_name)
            ->greeting('Hello ' . $notifiable->user_full_name . '!')
            ->line('Your WhatsApp verification code for **' . $business_name . '** is:')
            ->line('## **' . $this->verification_code . '**')
            ->line('This code will expire in **10 minutes**.')
            ->line('Please enter this code on the verification page to complete your account setup.')
            ->line('**Note:** This code was sent to WhatsApp number: ' . $this->contact_number)
            ->line('If you did not request this code, please ignore this email.')
            ->salutation('Best regards,  
' . $app_name . ' Team');
    }

    /**
     * Get the WhatsApp representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return string
     */
    public function toWhatsApp($notifiable)
    {
        $app_name = config('app.name', 'POS System');
        $business_name = $this->business->name;
        
        $message = "🔐 *WhatsApp Verification Code*\n\n";
        $message .= "Hello {$notifiable->user_full_name},\n\n";
        $message .= "Your verification code for *{$business_name}* is:\n\n";
        $message .= "```{$this->verification_code}```\n\n";
        $message .= "⏰ This code will expire in *10 minutes*.\n\n";
        $message .= "Please enter this code on the verification page to complete your account setup.\n\n";
        $message .= "🚫 If you did not request this code, please ignore this message.\n\n";
        $message .= "_{$app_name} Team_";
        
        return $message;
    }

    /**
     * Get the SMS representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return string
     */
    public function toNexmo($notifiable)
    {
        $app_name = config('app.name', 'POS');
        $business_name = $this->business->name;
        
        return "{$app_name}: Your verification code for {$business_name} is: {$this->verification_code}. Valid for 10 minutes.";
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        return [
            'business_id' => $this->business->id,
            'business_name' => $this->business->name,
            'verification_code' => $this->verification_code,
            'contact_number' => $this->contact_number,
        ];
    }
}
