<?php

namespace Modules\Superadmin\Notifications\Channels;

use App\Services\WhatsAppService;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Log;

class WhatsAppChannel
{
    protected $whatsappService;

    public function __construct(WhatsAppService $whatsappService)
    {
        $this->whatsappService = $whatsappService;
    }

    /**
     * Send the given notification.
     *
     * @param  mixed  $notifiable
     * @param  \Illuminate\Notifications\Notification  $notification
     * @return void
     */
    public function send($notifiable, Notification $notification)
    {
        try {
            // Check if WhatsApp is enabled
            if (!config('services.whatsapp.enabled', false)) {
                Log::warning('WhatsApp notification skipped: WhatsApp is disabled', [
                    'notifiable_id' => $notifiable->id ?? null,
                    'notification' => get_class($notification)
                ]);
                return;
            }

            // Get the phone number from the notifiable entity
            $phoneNumber = $this->getPhoneNumber($notifiable);
            
            if (empty($phoneNumber)) {
                Log::warning('WhatsApp notification skipped: No phone number', [
                    'notifiable_id' => $notifiable->id ?? null,
                    'notification' => get_class($notification)
                ]);
                return;
            }

            Log::info('WhatsApp channel processing', [
                'raw_phone' => $phoneNumber,
                'notifiable_id' => $notifiable->id ?? null,
                'notification' => get_class($notification)
            ]);

            // Format phone number to international format
            $phoneNumber = $this->whatsappService->formatPhoneNumber($phoneNumber);

            Log::info('WhatsApp phone formatted', [
                'formatted_phone' => $phoneNumber
            ]);

            // Check if the notification has a toWhatsApp method
            if (!method_exists($notification, 'toWhatsApp')) {
                Log::warning('WhatsApp notification skipped: toWhatsApp method missing', [
                    'notification' => get_class($notification)
                ]);
                return;
            }

            // Get the message content
            $message = $notification->toWhatsApp($notifiable);
            
            if (empty($message)) {
                Log::warning('WhatsApp notification skipped: Empty message', [
                    'notification' => get_class($notification)
                ]);
                return;
            }

            Log::info('WhatsApp message prepared', [
                'message_length' => strlen($message),
                'phone' => $phoneNumber
            ]);

            // Attempt to send via WhatsApp
            $result = $this->whatsappService->sendMessage($phoneNumber, $message);
            
            Log::info('WhatsApp service response', [
                'phone' => $phoneNumber,
                'result' => $result
            ]);
            
            if (!$result['success']) {
                Log::error('WhatsApp notification failed to send', [
                    'phone' => $phoneNumber,
                    'notification' => get_class($notification),
                    'error' => $result['error'] ?? 'Unknown error',
                    'response' => $result['response'] ?? null,
                    'http_code' => $result['http_code'] ?? null
                ]);
            } else {
                Log::info('WhatsApp notification sent successfully', [
                    'phone' => $phoneNumber,
                    'notification' => get_class($notification),
                    'response' => $result['response'] ?? null
                ]);
            }
        } catch (\Throwable $e) {
            // Catch any exception/error and log it, but don't rethrow
            // This ensures other notification channels continue to work
            Log::error('WhatsApp channel exception: ' . $e->getMessage(), [
                'notification' => get_class($notification),
                'phone' => $phoneNumber ?? 'unknown',
                'line' => $e->getLine(),
                'file' => $e->getFile(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Get the phone number from the notifiable entity
     *
     * @param  mixed  $notifiable
     * @return string|null
     */
    protected function getPhoneNumber($notifiable)
    {
        // First, try to get phone from business location (primary contact for business)
        if (isset($notifiable->business_id)) {
            try {
                $business = \App\Business::find($notifiable->business_id);
                if ($business && $business->locations) {
                    $location = $business->locations->first();
                    if ($location && !empty($location->mobile)) {
                        return $location->mobile;
                    }
                }
            } catch (\Exception $e) {
                Log::warning('Could not fetch business location phone', [
                    'business_id' => $notifiable->business_id,
                    'error' => $e->getMessage()
                ]);
            }
        }
        
        // Fallback to user's direct phone number fields
        $phoneFields = ['contact_no', 'contact_number', 'mobile', 'phone', 'phone_number'];
        
        foreach ($phoneFields as $field) {
            if (isset($notifiable->$field) && !empty($notifiable->$field)) {
                return $notifiable->$field;
            }
        }
        
        return null;
    }
}
