<?php

namespace Modules\Superadmin\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Messages\BroadcastMessage;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Modules\Superadmin\Notifications\Channels\WhatsAppChannel;

class SendSubscriptionExpiryAlert extends Notification
{
    use Queueable;

    protected $subscription;

    protected $days_left;

    /**
     * Create a new notification instance.
     *
     * @return void
     */
    public function __construct($subscription)
    {
        $this->subscription = $subscription;
        $this->days_left = \Carbon::now()->diffInDays($this->subscription->end_date) + 1;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        $channels = ['mail', 'database'];
        if (isPusherEnabled()) {
            $channels[] = 'broadcast';
        }
        
        if (config('services.whatsapp.enabled', false)) {
            $channels[] = WhatsAppChannel::class;
        }

        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        $owner_name = $notifiable->user_full_name;
        $app_name = config('app.name');
        $business_name = $this->subscription->business->name;
        $days_left = $this->days_left;

        return (new MailMessage)
                ->greeting("Dear $owner_name,")
                ->subject('Subscription Expiry Alert')
                ->line("Your subscription for $app_name is expiring in next $days_left days.")
                ->line("Kindly subscribe to continue using $business_name.")
                ->action('Subscribe', action([\Modules\Superadmin\Http\Controllers\SubscriptionController::class, 'index']));
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toDatabase($notifiable)
    {
        return [
            'days_left' => $this->days_left,
        ];
    }

    /**
     * Get the broadcastable representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return BroadcastMessage
     */
    public function toBroadcast($notifiable)
    {
        return new BroadcastMessage([
            'title' => '',
            'body' => strip_tags(__('superadmin::lang.subscription_expiry_alert', ['days_left' => $this->days_left, 'app_name' => config('app.name')])),
            'link' => action([\Modules\Superadmin\Http\Controllers\SubscriptionController::class, 'index']),
        ]);
    }

    /**
     * Get the WhatsApp representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return string
     */
    public function toWhatsApp($notifiable)
    {
        $message = "⏰ *Subscription Expiring Soon*\n\n";
        $message .= "Hello {$notifiable->user_full_name},\n\n";
        $message .= "Your subscription will expire in *{$this->days_left} days* on " . $this->subscription->end_date->format('M d, Y') . ".\n\n";
        $message .= "To continue enjoying our services without interruption, please renew your subscription.\n\n";
        $message .= "Log in to your account to manage your subscription.\n\n";
        $message .= "_" . config('app.name') . " Team_";
        
        return $message;
    }
}
