@extends('layouts.app')
@section('title', __('superadmin::lang.superadmin') . ' | Custom Domain Management')

@section('content')
    @include('superadmin::layouts.nav')
    
    <!-- Content Header (Page header) -->
    <section class="content-header">
        <h1 class="tw-text-xl md:tw-text-3xl tw-font-bold tw-text-black">
            <i class="fa fa-globe"></i> Custom Domain Management
            <small class="tw-text-sm md:tw-text-base tw-text-gray-700 tw-font-semibold">Approve or reject custom domain requests</small>
        </h1>
    </section>

    <!-- Main content -->
    <section class="content">
        <div class="tw-transition-all lg:tw-col-span-1 tw-duration-200 tw-bg-white tw-shadow-sm tw-rounded-xl tw-ring-1 hover:tw-shadow-md tw-ring-gray-200">
            <div class="tw-p-4 sm:tw-p-5">
                
                <!-- Pending Domains Section -->
                <div class="row">
                    <div class="col-md-12">
                        <h3 class="tw-font-bold tw-text-lg tw-mb-4">
                            <i class="fa fa-clock-o text-warning"></i> Pending Domain Verifications
                        </h3>
                        
                        <div id="pending-domains-container">
                            <div class="text-center">
                                <i class="fa fa-spinner fa-spin fa-2x"></i>
                                <p>Loading pending domains...</p>
                            </div>
                        </div>
                    </div>
                </div>

                <hr class="tw-my-6">

                <!-- All Businesses with Custom Domains -->
                <div class="row">
                    <div class="col-md-12">
                        <h3 class="tw-font-bold tw-text-lg tw-mb-4">
                            <i class="fa fa-check-circle text-success"></i> All Custom Domains
                        </h3>
                        
                        <div class="table-responsive">
                            <table class="table table-bordered table-striped" id="custom_domains_table">
                                <thead>
                                    <tr>
                                        <th>Business Name</th>
                                        <th>Custom Domain</th>
                                        <th>Status</th>
                                        <th>Verified At</th>
                                        <th>Business URL</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                            </table>
                        </div>
                    </div>
                </div>

            </div>
        </div>
    </section>

@endsection

@section('javascript')
<script type="text/javascript">
$(document).ready(function() {
    
    // Load pending domains
    loadPendingDomains();
    
    function loadPendingDomains() {
        $.ajax({
            url: "{{ route('superadmin.domains.pending') }}",
            type: 'GET',
            success: function(response) {
                if (response.success && response.data.length > 0) {
                    renderPendingDomains(response.data);
                } else {
                    $('#pending-domains-container').html(
                        '<div class="alert alert-info">' +
                        '<i class="fa fa-info-circle"></i> No pending domain verifications.' +
                        '</div>'
                    );
                }
            },
            error: function() {
                $('#pending-domains-container').html(
                    '<div class="alert alert-danger">' +
                    '<i class="fa fa-exclamation-circle"></i> Failed to load pending domains.' +
                    '</div>'
                );
            }
        });
    }
    
    function renderPendingDomains(domains) {
        let html = '<div class="row">';
        
        domains.forEach(function(domain) {
            html += `
                <div class="col-md-6 col-lg-4 mb-3">
                    <div class="box box-warning">
                        <div class="box-header with-border">
                            <h4 class="box-title">
                                <i class="fa fa-building"></i> ${domain.name}
                            </h4>
                        </div>
                        <div class="box-body">
                            <p><strong>Domain:</strong> <code>${domain.custom_domain}</code></p>
                            <p><strong>Requested:</strong> ${new Date(domain.created_at).toLocaleDateString()}</p>
                            
                            <div class="alert alert-info">
                                <small><i class="fa fa-info-circle"></i> Verify DNS is configured before approving</small>
                            </div>
                        </div>
                        <div class="box-footer">
                            <button class="btn btn-success btn-sm verify-domain" data-id="${domain.id}" data-domain="${domain.custom_domain}">
                                <i class="fa fa-check"></i> Verify & Activate
                            </button>
                            <button class="btn btn-danger btn-sm reject-domain" data-id="${domain.id}">
                                <i class="fa fa-times"></i> Reject
                            </button>
                        </div>
                    </div>
                </div>
            `;
        });
        
        html += '</div>';
        $('#pending-domains-container').html(html);
    }
    
    // Verify domain
    $(document).on('click', '.verify-domain', function() {
        let businessId = $(this).data('id');
        let domain = $(this).data('domain');
        
        if (!confirm(`Are you sure you want to verify and activate the domain "${domain}"?\n\nPlease ensure DNS is properly configured.`)) {
            return;
        }
        
        let btn = $(this);
        btn.prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Verifying...');
        
        $.ajax({
            url: `/superadmin/custom-domains/${businessId}/verify`,
            type: 'POST',
            data: {
                _token: '{{ csrf_token() }}'
            },
            success: function(response) {
                if (response.success) {
                    toastr.success(response.msg);
                    loadPendingDomains();
                    $('#custom_domains_table').DataTable().ajax.reload();
                } else {
                    toastr.error(response.msg);
                    btn.prop('disabled', false).html('<i class="fa fa-check"></i> Verify & Activate');
                }
            },
            error: function(xhr) {
                let msg = xhr.responseJSON?.msg || 'Failed to verify domain';
                toastr.error(msg);
                btn.prop('disabled', false).html('<i class="fa fa-check"></i> Verify & Activate');
            }
        });
    });
    
    // Reject domain
    $(document).on('click', '.reject-domain', function() {
        let businessId = $(this).data('id');
        
        let reason = prompt('Enter reason for rejection (optional):');
        if (reason === null) return; // User cancelled
        
        let btn = $(this);
        btn.prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Rejecting...');
        
        $.ajax({
            url: `/superadmin/custom-domains/${businessId}/reject`,
            type: 'POST',
            data: {
                _token: '{{ csrf_token() }}',
                reason: reason
            },
            success: function(response) {
                if (response.success) {
                    toastr.success(response.msg);
                    loadPendingDomains();
                    $('#custom_domains_table').DataTable().ajax.reload();
                } else {
                    toastr.error(response.msg);
                    btn.prop('disabled', false).html('<i class="fa fa-times"></i> Reject');
                }
            },
            error: function(xhr) {
                let msg = xhr.responseJSON?.msg || 'Failed to reject domain';
                toastr.error(msg);
                btn.prop('disabled', false).html('<i class="fa fa-times"></i> Reject');
            }
        });
    });
    
    // DataTable for all custom domains
    $('#custom_domains_table').DataTable({
        processing: true,
        ajax: {
            url: "{{ action([\Modules\Superadmin\Http\Controllers\BusinessController::class, 'index']) }}",
            dataSrc: function(json) {
                // Extract businesses with custom domains from the DataTables response
                let allData = json.data || [];
                return allData.filter(function(business) {
                    return business.custom_domain;
                });
            }
        },
        columns: [
            { 
                data: 'name',
                render: function(data) {
                    return data || '-';
                }
            },
            { 
                data: 'custom_domain',
                render: function(data) {
                    return data ? `<code>${data}</code>` : '-';
                }
            },
            { 
                data: null,
                render: function(data) {
                    if (data.custom_domain_enabled && data.custom_domain_verified_at) {
                        return '<span class="label label-success"><i class="fa fa-check-circle"></i> Verified</span>';
                    } else if (data.custom_domain) {
                        return '<span class="label label-warning"><i class="fa fa-clock-o"></i> Pending</span>';
                    }
                    return '<span class="label label-default">-</span>';
                }
            },
            { 
                data: 'custom_domain_verified_at',
                render: function(data) {
                    return data ? new Date(data).toLocaleString() : '-';
                }
            },
            { 
                data: null,
                render: function(data) {
                    if (data.custom_domain_enabled && data.custom_domain) {
                        let protocol = '{{ config("app.env") }}' === 'production' ? 'https://' : 'http://';
                        let url = protocol + data.custom_domain;
                        return `<a href="${url}" target="_blank">${url} <i class="fa fa-external-link"></i></a>`;
                    }
                    return '-';
                }
            },
            { 
                data: null,
                orderable: false,
                searchable: false,
                render: function(data) {
                    let actions = '';
                    if (!data.custom_domain_enabled && data.custom_domain) {
                        actions += `<button class="btn btn-xs btn-success verify-domain" data-id="${data.id}" data-domain="${data.custom_domain}">
                                        <i class="fa fa-check"></i> Verify
                                    </button> `;
                    }
                    if (data.custom_domain) {
                        actions += `<button class="btn btn-xs btn-danger reject-domain" data-id="${data.id}">
                                        <i class="fa fa-times"></i> Remove
                                    </button>`;
                    }
                    return actions || '-';
                }
            }
        ],
        order: [[3, 'desc']], // Sort by verified_at descending
        pageLength: 25
    });
    
});
</script>
@endsection
