<?php

namespace Modules\Superadmin\Services;

use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class WhatsAppTicketService
{
    protected $apiUrl;
    protected $apiToken;
    protected $deviceId;

    public function __construct()
    {
        $this->apiUrl = config('services.whatsapp_ticket.api_url', 'https://nearbuymarts.com/api/waticket');
        $this->apiToken = config('services.whatsapp_ticket.api_token', env('WHATSAPP_TICKET_API_TOKEN'));
        $this->deviceId = config('services.whatsapp_ticket.device_id', env('WHATSAPP_TICKET_DEVICE_ID', '123'));
    }

    /**
     * Create a new ticket via WhatsApp API
     */
    public function createTicket($data)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiToken,
                'X-Device-ID' => $this->deviceId,
                'Content-Type' => 'application/json',
            ])->post($this->apiUrl . '/tickets', [
                'customer_phone' => $data['customer_phone'],
                'customer_name' => $data['customer_name'],
                'subject' => $data['subject'],
                'description' => $data['description'],
                'priority' => $data['priority'] ?? 'medium',
                'category' => $data['category'] ?? 'general_inquiry',
                'external_reference' => $data['external_reference'] ?? null,
            ]);

            if ($response->successful()) {
                $result = $response->json();
                
                if ($result['success']) {
                    Log::info('WhatsApp ticket created successfully', [
                        'ticket_number' => $result['data']['ticket_number'] ?? null
                    ]);
                    
                    return [
                        'success' => true,
                        'ticket_number' => $result['data']['ticket_number'] ?? null,
                        'data' => $result['data'] ?? null,
                    ];
                }
            }

            Log::warning('WhatsApp ticket creation failed', [
                'status' => $response->status(),
                'response' => $response->body()
            ]);

            return [
                'success' => false,
                'error' => 'Failed to create WhatsApp ticket',
                'details' => $response->json(),
            ];

        } catch (Exception $e) {
            Log::error('WhatsApp ticket creation exception', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Send a message to an existing ticket
     */
    public function sendMessage($ticketNumber, $message)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiToken,
                'X-Device-ID' => $this->deviceId,
                'Content-Type' => 'application/json',
            ])->post($this->apiUrl . '/tickets/' . $ticketNumber . '/messages', [
                'message' => $message,
            ]);

            if ($response->successful()) {
                $result = $response->json();
                
                return [
                    'success' => true,
                    'data' => $result['data'] ?? null,
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to send message',
            ];

        } catch (Exception $e) {
            Log::error('WhatsApp message send exception', [
                'ticket_number' => $ticketNumber,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Update ticket status
     */
    public function updateTicketStatus($ticketNumber, $status)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiToken,
                'X-Device-ID' => $this->deviceId,
                'Content-Type' => 'application/json',
            ])->patch($this->apiUrl . '/tickets/' . $ticketNumber, [
                'status' => $status,
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json(),
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to update ticket status',
            ];

        } catch (Exception $e) {
            Log::error('WhatsApp ticket status update exception', [
                'ticket_number' => $ticketNumber,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Get ticket details from WhatsApp API
     */
    public function getTicket($ticketNumber)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiToken,
                'X-Device-ID' => $this->deviceId,
            ])->get($this->apiUrl . '/tickets/' . $ticketNumber);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json(),
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to get ticket details',
            ];

        } catch (Exception $e) {
            Log::error('WhatsApp get ticket exception', [
                'ticket_number' => $ticketNumber,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Check if WhatsApp ticket service is enabled
     */
    public function isEnabled()
    {
        return !empty($this->apiToken) && !empty($this->apiUrl);
    }

    /**
     * Process incoming WhatsApp message and auto-create ticket if keywords detected
     * 
     * @param array $data Webhook payload from WhatsApp API
     * @return array Result of processing
     */
    public function processIncomingMessage($data)
    {
        try {
            // Extract message data
            $phoneNumber = $data['from'] ?? null;
            $message = $data['message'] ?? null;
            $senderName = $data['sender_name'] ?? null;

            if (!$phoneNumber || !$message) {
                return [
                    'success' => false,
                    'error' => 'Missing required fields: phone number or message',
                ];
            }

            // Check if message contains ticket keywords
            $ticketKeywords = $this->getTicketKeywords();
            $isTicketRequest = false;
            $detectedCategory = 'general_inquiry';
            $priority = 'medium';

            foreach ($ticketKeywords as $category => $keywords) {
                foreach ($keywords as $keyword) {
                    if (stripos($message, $keyword) !== false) {
                        $isTicketRequest = true;
                        $detectedCategory = $category;
                        break 2;
                    }
                }
            }

            if (!$isTicketRequest) {
                return [
                    'success' => false,
                    'message' => 'No ticket keywords detected in message',
                ];
            }

            // Auto-detect priority based on urgency keywords
            $priority = $this->detectPriority($message);

            // Find business by phone number
            $business = $this->findBusinessByPhone($phoneNumber);

            if (!$business) {
                // Send notification to admin about unrecognized phone number
                $this->notifyAdminAboutUnknownSender($phoneNumber, $message, $senderName);
                
                return [
                    'success' => false,
                    'error' => 'Business not found for phone number',
                    'notification_sent' => true,
                ];
            }

            // Auto-create support ticket
            $ticket = $this->autoCreateTicket($business, $phoneNumber, $message, $detectedCategory, $priority);

            if ($ticket) {
                // Send confirmation message back to user
                $this->sendAutoReply($phoneNumber, $ticket, $business);

                return [
                    'success' => true,
                    'ticket' => $ticket,
                    'message' => 'Ticket created successfully from WhatsApp message',
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to create ticket',
            ];

        } catch (Exception $e) {
            Log::error('WhatsApp message processing exception', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Get support ticket keywords grouped by category
     */
    protected function getTicketKeywords()
    {
        return [
            'billing' => ['billing', 'payment', 'invoice', 'subscription', 'charge', 'refund'],
            'technical_support' => ['help', 'error', 'bug', 'issue', 'problem', 'not working', 'broken', 'fix'],
            'feature_request' => ['feature', 'request', 'add', 'new feature', 'suggestion', 'improve'],
            'bug_report' => ['bug', 'crash', 'freeze', 'error message', 'exception'],
            'account_issue' => ['account', 'login', 'password', 'access', 'locked', 'suspended'],
            'subscription' => ['package', 'plan', 'upgrade', 'downgrade', 'renew', 'expiry'],
        ];
    }

    /**
     * Detect message priority based on urgency keywords
     */
    protected function detectPriority($message)
    {
        $urgentKeywords = ['urgent', 'emergency', 'critical', 'asap', 'immediately', 'down', 'crash'];
        $highKeywords = ['important', 'high priority', 'need help', 'stuck'];

        foreach ($urgentKeywords as $keyword) {
            if (stripos($message, $keyword) !== false) {
                return 'urgent';
            }
        }

        foreach ($highKeywords as $keyword) {
            if (stripos($message, $keyword) !== false) {
                return 'high';
            }
        }

        return 'medium';
    }

    /**
     * Find business by WhatsApp phone number
     */
    protected function findBusinessByPhone($phoneNumber)
    {
        // Clean phone number (remove spaces, dashes, etc.)
        $cleanPhone = preg_replace('/[^0-9]/', '', $phoneNumber);

        // Try to find business owner by contact number
        $user = \App\User::whereRaw('REPLACE(REPLACE(REPLACE(contact_number, " ", ""), "-", ""), "+", "") LIKE ?', ["%{$cleanPhone}%"])
            ->whereHas('business')
            ->first();

        if ($user && $user->business_id) {
            return \App\Business::find($user->business_id);
        }

        // Also try matching business contact_number field if it exists
        $business = \App\Business::whereRaw('REPLACE(REPLACE(REPLACE(contact_number, " ", ""), "-", ""), "+", "") LIKE ?', ["%{$cleanPhone}%"])
            ->first();

        return $business;
    }

    /**
     * Auto-create support ticket from WhatsApp message
     */
    protected function autoCreateTicket($business, $phoneNumber, $message, $category, $priority)
    {
        // Generate unique ticket number
        $ticketNumber = 'TKT-' . strtoupper(uniqid());

        // Extract subject from first line or use category as subject
        $lines = explode("\n", $message);
        $subject = isset($lines[0]) && strlen($lines[0]) > 0 
            ? substr($lines[0], 0, 100) 
            : ucfirst(str_replace('_', ' ', $category));

        // Create ticket
        $ticket = \Modules\Superadmin\Entities\SupportTicket::create([
            'business_id' => $business->id,
            'ticket_number' => $ticketNumber,
            'subject' => $subject,
            'category' => $category,
            'priority' => $priority,
            'description' => $message,
            'status' => 'open',
            'created_by' => $business->owner->id ?? null,
            'source' => 'whatsapp',
            'whatsapp_phone' => $phoneNumber,
        ]);

        // Create initial system message
        \Modules\Superadmin\Entities\SupportTicketMessage::create([
            'ticket_id' => $ticket->id,
            'user_id' => $business->owner->id ?? null,
            'message' => "Ticket auto-created from WhatsApp message (Phone: {$phoneNumber})",
            'sender_type' => 'system',
            'is_system_message' => true,
        ]);

        // Notify superadmin about new auto-created ticket
        $this->notifyAdminAboutNewTicket($ticket, $business);

        return $ticket;
    }

    /**
     * Send auto-reply confirmation to user
     */
    protected function sendAutoReply($phoneNumber, $ticket, $business)
    {
        $adminPhone = \App\System::getProperty('whatsapp_admin_phone');
        
        if (!$adminPhone) {
            return;
        }

        $message = "Hi {$business->owner->user_full_name},\n\n";
        $message .= "Your support ticket has been created successfully!\n\n";
        $message .= "Ticket Number: {$ticket->ticket_number}\n";
        $message .= "Subject: {$ticket->subject}\n";
        $message .= "Priority: " . ucfirst($ticket->priority) . "\n";
        $message .= "Status: " . ucfirst($ticket->status) . "\n\n";
        $message .= "Our support team will respond to you shortly.\n";
        $message .= "You can also view your ticket in your dashboard.";

        try {
            // Use simple HTTP to send message via NearBuyMarts API
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiToken,
                'Content-Type' => 'application/json',
            ])->post($this->apiUrl . '/send', [
                'phone' => $phoneNumber,
                'message' => $message,
            ]);

            if ($response->successful()) {
                Log::info('Auto-reply sent successfully', ['ticket' => $ticket->ticket_number]);
            }
        } catch (Exception $e) {
            Log::warning('Failed to send auto-reply: ' . $e->getMessage());
        }
    }

    /**
     * Notify admin about new auto-created ticket
     */
    protected function notifyAdminAboutNewTicket($ticket, $business)
    {
        $adminPhone = \App\System::getProperty('whatsapp_admin_phone');
        
        if (!$adminPhone) {
            return;
        }

        $message = "🎫 New Support Ticket Auto-Created\n\n";
        $message .= "Ticket: {$ticket->ticket_number}\n";
        $message .= "Business: {$business->name}\n";
        $message .= "Subject: {$ticket->subject}\n";
        $message .= "Category: " . ucfirst(str_replace('_', ' ', $ticket->category)) . "\n";
        $message .= "Priority: " . ucfirst($ticket->priority) . "\n";
        $message .= "Source: WhatsApp Auto-Creation\n\n";
        $message .= "View: " . route('superadmin.support-tickets.show', $ticket->id);

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiToken,
                'Content-Type' => 'application/json',
            ])->post($this->apiUrl . '/send', [
                'phone' => $adminPhone,
                'message' => $message,
            ]);

            if ($response->successful()) {
                Log::info('Admin notified about auto-created ticket', ['ticket' => $ticket->ticket_number]);
            }
        } catch (Exception $e) {
            Log::warning('Failed to notify admin: ' . $e->getMessage());
        }
    }

    /**
     * Notify admin about message from unknown sender
     */
    protected function notifyAdminAboutUnknownSender($phoneNumber, $message, $senderName)
    {
        $adminPhone = \App\System::getProperty('whatsapp_admin_phone');
        
        if (!$adminPhone) {
            return;
        }

        $notification = "⚠️ Message from Unknown Sender\n\n";
        $notification .= "Phone: {$phoneNumber}\n";
        $notification .= "Name: " . ($senderName ?? 'Unknown') . "\n\n";
        $notification .= "Message:\n{$message}\n\n";
        $notification .= "This phone number is not associated with any business in the system.";

        try {
            Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiToken,
                'Content-Type' => 'application/json',
            ])->post($this->apiUrl . '/send', [
                'phone' => $adminPhone,
                'message' => $notification,
            ]);
        } catch (Exception $e) {
            Log::warning('Failed to notify admin about unknown sender: ' . $e->getMessage());
        }
    }
}
