<?php

namespace Modules\WhatsAppBot\Http\Controllers;

use App\Utils\ModuleUtil;
use Illuminate\Routing\Controller;
use Nwidart\Menus\Facades\Menu;

class DataController extends Controller
{
    /**
     * Defines module as a superadmin package.
     * @return array
     */
    public function superadmin_package()
    {
        return [
            [
                'name' => 'whatsapp_bot_module',
                'label' => __('whatsappbot::lang.whatsapp_bot_module'),
                'default' => false
            ]
        ];
    }

    /**
     * Defines user permissions for the module.
     * @return array
     */
    public function user_permissions()
    {
        return [
            [
                'value' => 'whatsappbot.access',
                'label' => __('whatsappbot::lang.access_whatsapp_bot'),
                'default' => false
            ],
            [
                'value' => 'whatsappbot.view_dashboard',
                'label' => __('whatsappbot::lang.view_dashboard'),
                'default' => false
            ],
            [
                'value' => 'whatsappbot.manage_settings',
                'label' => __('whatsappbot::lang.manage_settings'),
                'default' => false
            ],
            [
                'value' => 'whatsappbot.send_messages',
                'label' => __('whatsappbot::lang.send_messages'),
                'default' => false
            ],
            [
                'value' => 'whatsappbot.view_logs',
                'label' => __('whatsappbot::lang.view_logs'),
                'default' => false
            ],
        ];
    }

    /**
     * Adds WhatsAppBot menus
     * @return null
     */
    public function modifyAdminMenu()
    {
        $business_id = session()->get('user.business_id');
        $module_util = new ModuleUtil();
        $is_whatsapp_bot_enabled = (boolean)$module_util->hasThePermissionInSubscription($business_id, 'whatsapp_bot_module', 'superadmin_package');

        $background_color = '#25D366'; // WhatsApp green color
        if (config('app.env') == 'demo') {
            $background_color = '#25D366 !important;color:white';
        }

        if ($is_whatsapp_bot_enabled && (auth()->user()->can('superadmin') || auth()->user()->can('whatsappbot.access'))) {
            
            Menu::modify('admin-sidebar-menu', function ($menu) use ($background_color) {
                $menu->dropdown(__('whatsappbot::lang.whatsapp_bot'), 
                    
                    function ($sub) use ($background_color) {
                        // Dashboard
                        if (auth()->user()->can('superadmin') || auth()->user()->can('whatsappbot.view_dashboard')) {
                            $sub->url(
                                action([\Modules\WhatsAppBot\Http\Controllers\WhatsAppBotController::class, 'index']),
                                __('whatsappbot::lang.dashboard'),
                                ['icon' => '', 'active' => request()->segment(1) == 'whatsappbot' && (request()->segment(2) == '' || request()->segment(2) == 'index')]
                            );
                        }

                        // Settings
                        if (auth()->user()->can('superadmin') || auth()->user()->can('whatsappbot.manage_settings')) {
                            $sub->url(
                                action([\Modules\WhatsAppBot\Http\Controllers\WhatsAppBotController::class, 'settings']),
                                __('whatsappbot::lang.settings'),
                                ['icon' => '', 'active' => request()->segment(1) == 'whatsappbot' && request()->segment(2) == 'settings']
                            );
                        }

                        // Message Logs
                        if (auth()->user()->can('superadmin') || auth()->user()->can('whatsappbot.view_logs')) {
                            $sub->url(
                                action([\Modules\WhatsAppBot\Http\Controllers\WhatsAppBotController::class, 'logs']),
                                __('whatsappbot::lang.logs'),
                                ['icon' => '', 'active' => request()->segment(1) == 'whatsappbot' && request()->segment(2) == 'logs']
                            );
                        }

                        // Test Messages
                        if (auth()->user()->can('superadmin') || auth()->user()->can('whatsappbot.send_messages')) {
                            $sub->url(
                                action([\Modules\WhatsAppBot\Http\Controllers\WhatsAppBotController::class, 'testMessages']),
                                __('whatsappbot::lang.test_messages'),
                                ['icon' => '', 'active' => request()->segment(1) == 'whatsappbot' && request()->segment(2) == 'test-messages']
                            );
                        }
                    },
                    [
                        'icon' => '<svg aria-hidden="true" class="tw-size-5 tw-shrink-0" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                            <path stroke="none" d="M0 0h24v24H0z" fill="none"></path>
                            <path d="M3 21l1.65 -3.8a9 9 0 1 1 3.4 2.9l-5.05 .9"></path>
                            <path d="M9 10a.5 .5 0 0 0 1 0v-1a.5 .5 0 0 0 -1 0v1a5 5 0 0 0 5 5h1a.5 .5 0 0 0 0 -1h-1a.5 .5 0 0 0 0 1"></path>
                        </svg>',
                        'active' => request()->segment(1) == 'whatsappbot',
                        'style' => 'background-color:' . $background_color . ' !important;'
                    ]
                )->order(25);
            });
        }
    }

    /**
     * Function to be called after a transaction is saved.
     * Can be used to send WhatsApp notifications
     *
     * @param array $data
     * @return null
     */
    public function after_sale_saved($data)
    {
        try {
            $business_id = session()->get('user.business_id');
            $module_util = new ModuleUtil();
            $is_whatsapp_bot_enabled = (boolean)$module_util->hasThePermissionInSubscription($business_id, 'whatsapp_bot_module', 'superadmin_package');

            if (!$is_whatsapp_bot_enabled) {
                return;
            }

            $transaction = $data['transaction'];
            
            // Check if WhatsApp notifications are enabled for this business
            $business = \DB::table('business')->where('id', $business_id)->first();
            if (!$business || !$business->whatsapp_enabled || !$business->auto_send_invoices) {
                return;
            }

            // Get customer contact info
            $customer_phone = null;
            $customer_name = null;
            if ($transaction->contact_id) {
                $contact = \DB::table('contacts')->where('id', $transaction->contact_id)->first();
                if ($contact && !empty($contact->mobile)) {
                    $customer_phone = $contact->mobile;
                    $customer_name = $contact->name;
                }
            }

            // Only send invoice if customer has phone number and transaction is finalized
            if ($customer_phone && 
                !empty($business->whatsapp_phone) && 
                $transaction->status == 'final' && 
                $transaction->type == 'sell' &&
                !$transaction->is_quotation) {
                
                // Send invoice PDF via WhatsApp
                $whatsapp_service = app(\Modules\WhatsAppBot\Services\WhatsAppBotService::class);
                
                try {
                    // Generate PDF
                    $pdfPath = $whatsapp_service->generateInvoicePDF($transaction);
                    
                    if ($pdfPath) {
                        // Send invoice with PDF attachment
                        $response = $whatsapp_service->sendInvoicePDF($transaction, $customer_phone, $pdfPath);
                        
                        if ($response) {
                            \Log::info('WhatsAppBot: Invoice sent successfully', [
                                'transaction_id' => $transaction->id,
                                'invoice_no' => $transaction->invoice_no,
                                'customer_phone' => $customer_phone,
                                'customer_name' => $customer_name
                            ]);
                        }
                        
                        // Clean up temporary PDF file after sending
                        if (file_exists($pdfPath)) {
                            unlink($pdfPath);
                        }
                    } else {
                        // If PDF generation fails, send text-only notification
                        $message = "🧾 *Invoice from {$business->name}*\n\n";
                        $message .= "Invoice No: *{$transaction->invoice_no}*\n";
                        $message .= "Date: " . \Carbon\Carbon::parse($transaction->transaction_date)->format('d M Y, h:i A') . "\n";
                        $message .= "Amount: ₦" . number_format($transaction->final_total, 2) . "\n\n";
                        $message .= "Thank you for your business! 🙏\n\n";
                        $message .= "*{$business->name}*";

                        $whatsapp_service->sendTestMessage($customer_phone, $message);
                        
                        \Log::warning('WhatsAppBot: PDF generation failed, sent text notification', [
                            'transaction_id' => $transaction->id,
                            'invoice_no' => $transaction->invoice_no,
                            'customer_phone' => $customer_phone
                        ]);
                    }
                    
                } catch (\Exception $e) {
                    \Log::error('WhatsAppBot: Invoice send failed', [
                        'transaction_id' => $transaction->id,
                        'invoice_no' => $transaction->invoice_no,
                        'customer_phone' => $customer_phone,
                        'error' => $e->getMessage()
                    ]);
                }
            }
        } catch (\Exception $e) {
            \Log::error('WhatsAppBot after_sale_saved error: ' . $e->getMessage());
        }

        return null;
    }

    /**
     * Function to be called after a purchase is saved.
     * Can be used to send WhatsApp notifications to suppliers
     *
     * @param array $data
     * @return null
     */
    public function after_purchase_saved($data)
    {
        try {
            $business_id = session()->get('user.business_id');
            $module_util = new ModuleUtil();
            $is_whatsapp_bot_enabled = (boolean)$module_util->hasThePermissionInSubscription($business_id, 'whatsapp_bot_module', 'superadmin_package');

            if (!$is_whatsapp_bot_enabled) {
                return;
            }

            $transaction = $data['transaction'];
            
            // Check if WhatsApp notifications are enabled for this business
            $business = \DB::table('business')->where('id', $business_id)->first();
            if (!$business || !$business->whatsapp_enabled) {
                return;
            }

            // Get supplier contact info
            $supplier_phone = null;
            if ($transaction->contact_id) {
                $contact = \DB::table('contacts')->where('id', $transaction->contact_id)->first();
                if ($contact && !empty($contact->mobile)) {
                    $supplier_phone = $contact->mobile;
                }
            }

            if ($supplier_phone && !empty($business->whatsapp_phone)) {
                // Send purchase notification via WhatsApp
                $whatsapp_service = app(\Modules\WhatsAppBot\Services\WhatsAppBotService::class);
                
                $message = "Purchase Order Confirmation\n\n";
                $message .= "PO Number: " . $transaction->ref_no . "\n";
                $message .= "Total: " . number_format($transaction->final_total, 2) . "\n";
                $message .= "Date: " . \Carbon\Carbon::parse($transaction->transaction_date)->format('d/m/Y') . "\n\n";
                $message .= "From: " . $business->name;

                try {
                    $whatsapp_service->sendTestMessage($supplier_phone, $message);
                } catch (\Exception $e) {
                    \Log::warning('WhatsApp purchase notification failed: ' . $e->getMessage());
                }
            }
        } catch (\Exception $e) {
            \Log::error('WhatsAppBot after_purchase_saved error: ' . $e->getMessage());
        }

        return null;
    }

    /**
     * Function to parse notification messages for WhatsApp Bot
     *
     * @param object $notification
     * @return array
     */
    public function parse_notification($notification)
    {
        $notification_data = [];
        
        // Handle WhatsApp Bot specific notifications
        if ($notification->type == 'Modules\WhatsAppBot\Notifications\WhatsAppMessageReceived') {
            $data = $notification->data;
            $msg = __('whatsappbot::lang.new_whatsapp_message', [
                'phone' => $data['phone'] ?? 'Unknown',
                'message' => \Str::limit($data['message'] ?? '', 50)
            ]);

            $notification_data = [
                'msg' => $msg,
                'icon_class' => 'fas fa-comment bg-green',
                'link' => action([\Modules\WhatsAppBot\Http\Controllers\WhatsAppBotController::class, 'index']),
                'read_at' => $notification->read_at,
                'created_at' => $notification->created_at->diffForHumans(),
            ];
        }

        return $notification_data;
    }
}