<?php

namespace Modules\WhatsAppBot\Http\Controllers;

use App\System;
use Composer\Semver\Comparator;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Exception;

/**
 * WhatsAppBot Module Installation Controller
 * Handles installation, updates, and uninstallation of the module
 */
class InstallController extends Controller
{
    protected $module_name;
    protected $appVersion;
    protected $module_display_name;

    public function __construct()
    {
        $this->module_name = 'whatsappbot';
        $this->appVersion = config('whatsappbot.module_version', '1.0.0');
        $this->module_display_name = 'WhatsAppBot';
    }

    /**
     * Display the installation page
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        ini_set('max_execution_time', 0);
        ini_set('memory_limit', '512M');

        $this->installSettings();

        //Check if installed or not.
        $is_installed = System::getProperty($this->module_name . '_version');
        if (!empty($is_installed)) {
            abort(404);
        }

        $action_url = action([\Modules\WhatsAppBot\Http\Controllers\InstallController::class, 'install']);

        $intruction_type = 'cc';
        $action_type = 'install';
        $module_display_name = $this->module_display_name;

        return view('install.install-module')
            ->with(compact('action_url', 'intruction_type', 'action_type', 'module_display_name'));
    }

    /**
     * Install the module
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function install(Request $request)
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            ini_set('max_execution_time', 0);
            ini_set('memory_limit', '512M');

            $this->installSettings();

            DB::beginTransaction();

            // Run migrations
            $exit_code = Artisan::call('module:migrate', ['module' => 'WhatsAppBot', '--force' => true]);

            // Add version to system properties
            System::addProperty($this->module_name . '_version', $this->appVersion);

            DB::commit();

            $output = ['success' => 1, 'msg' => $this->module_display_name . ' module installed successfully!'];
        } catch (Exception $e) {
            DB::rollBack();
            \Log::emergency('File: ' . $e->getFile() . ' Line: ' . $e->getLine() . ' Message: ' . $e->getMessage());
            $output = [
                'success' => 0,
                'msg' => $e->getMessage()
            ];
        }

        return redirect()
            ->action([\App\Http\Controllers\Install\ModulesController::class, 'index'])
            ->with('status', $output);
    }

    /**
     * Show update page
     *
     * @return \Illuminate\View\View
     */
    public function update()
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        ini_set('max_execution_time', 0);
        ini_set('memory_limit', '512M');

        $this->installSettings();

        // Check if WhatsAppBot is installed or not
        $version_installed = System::getProperty($this->module_name . '_version');
        if (empty($version_installed)) {
            abort(404);
        }

        $is_updated = Comparator::greaterThan($this->appVersion, $version_installed) ? false : true;

        if ($is_updated) {
            abort(404);
        }

        $action_url = action([\Modules\WhatsAppBot\Http\Controllers\InstallController::class, 'updateModule']);

        $intruction_type = 'iu';
        $action_type = 'update';
        $module_display_name = $this->module_display_name;

        return view('install.install-module')
            ->with(compact('action_url', 'intruction_type', 'action_type', 'module_display_name'));
    }

    /**
     * Initialize update
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateModule(Request $request)
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            ini_set('max_execution_time', 0);
            ini_set('memory_limit', '512M');

            $this->installSettings();

            DB::beginTransaction();

            // Run migrations
            Artisan::call('module:migrate', ['module' => 'WhatsAppBot', '--force' => true]);

            // Update version
            System::setProperty($this->module_name . '_version', $this->appVersion);

            DB::commit();

            $output = ['success' => 1, 'msg' => $this->module_display_name . ' module updated successfully to ' . $this->appVersion . '!'];
        } catch (Exception $e) {
            DB::rollBack();
            \Log::emergency('File: ' . $e->getFile() . ' Line: ' . $e->getLine() . ' Message: ' . $e->getMessage());
            $output = [
                'success' => 0,
                'msg' => $e->getMessage()
            ];
        }

        return redirect()
            ->action([\App\Http\Controllers\Install\ModulesController::class, 'index'])
            ->with('status', $output);
    }

    /**
     * Uninstall the module
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function uninstall(Request $request)
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            ini_set('max_execution_time', 0);
            ini_set('memory_limit', '512M');

            $this->installSettings();

            DB::beginTransaction();

            // Remove version from system
            System::removeProperty($this->module_name . '_version');

            // Rollback migrations (optional - uncomment if you want to drop tables)
            // $this->rollbackMigrations();

            DB::commit();

            $output = ['success' => 1, 'msg' => $this->module_display_name . ' module uninstalled successfully!'];
        } catch (Exception $e) {
            DB::rollBack();
            \Log::emergency('File: ' . $e->getFile() . ' Line: ' . $e->getLine() . ' Message: ' . $e->getMessage());
            $output = [
                'success' => 0,
                'msg' => $e->getMessage()
            ];
        }

        return redirect()
            ->action([\App\Http\Controllers\Install\ModulesController::class, 'index'])
            ->with('status', $output);
    }

    /**
     * Install default settings
     */
    private function installSettings()
    {
        $business_id = request()->session()->get('user.business_id');
        $user_id = request()->session()->get('user.id');
    }

    /**
     * Rollback module migrations (optional cleanup)
     */
    private function rollbackMigrations()
    {
        try {
            // Drop tables if they exist
            $tables = [
                'whatsapp_bot_logs',
            ];

            foreach ($tables as $table) {
                if (Schema::hasTable($table)) {
                    Schema::dropIfExists($table);
                }
            }

            // Remove whatsapp_enabled column from business table if it exists
            if (Schema::hasTable('business')) {
                Schema::table('business', function ($table) {
                    if (Schema::hasColumn('business', 'whatsapp_enabled')) {
                        $table->dropColumn('whatsapp_enabled');
                    }
                });
            }
        } catch (Exception $e) {
            \Log::error('WhatsAppBot Migration Rollback Error: ' . $e->getMessage());
        }
    }
}
