<?php

namespace Modules\WhatsAppBot\Http\Controllers;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\WhatsAppBot\Services\WhatsAppBotService;
use Illuminate\Support\Facades\Log;

class WhatsAppBotController extends Controller
{
    /**
     * @var WhatsAppBotService
     */
    protected $bot;

    /**
     * Create a new controller instance.
     *
     * @param WhatsAppBotService $bot
     */
    public function __construct(WhatsAppBotService $bot)
    {
        $this->bot = $bot;
    }

    /**
     * Display module dashboard.
     * @return Renderable
     */
    public function index()
    {
        $business_id = session()->get('user.business_id');
        $business = \DB::table('business')->where('id', $business_id)->first();
        $business_phone = $business->whatsapp_phone ?? null;
        
        return view('whatsappbot::index', compact('business_phone'));
    }

    /**
     * Handle incoming WhatsApp messages from NearBuyMarts webhook
     * 
     * Expected payload format:
     * {
     *   "sender": "+2348012345678",
     *   "message": "What's today's sales?",
     *   "timestamp": "2024-01-01 12:00:00"
     * }
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function handleIncoming(Request $request)
    {
        try {
            // Correlate requests for multi-tenant SaaS tracing
            $correlation_id = (string) \Illuminate\Support\Str::uuid();
            // Log ALL incoming requests (even before validation) for debugging
            Log::info('WhatsAppBot: RAW webhook request received', [
                'correlation_id' => $correlation_id,
                'ip' => $request->ip(),
                'method' => $request->method(),
                'user_agent' => $request->userAgent(),
                'headers' => [
                    'content_type' => $request->header('Content-Type'),
                    'has_webhook_token' => $request->hasHeader('X-Webhook-Token'),
                ],
                'has_body' => !empty($request->all()),
                'timestamp' => now()->toDateTimeString()
            ]);

            // SECURITY: Verify webhook token FIRST (mandatory, not optional)
            $verifyToken = config('whatsappbot.verify_webhook_token');
            if (empty($verifyToken)) {
                Log::critical('WhatsAppBot: Webhook token not configured - rejecting all requests');
                return response()->json([
                    'status' => 'error',
                    'message' => 'Webhook not configured'
                ], 503);
            }

            $requestToken = $request->header('X-Webhook-Token') ?? $request->input('token');
            
            if (!hash_equals($verifyToken, $requestToken ?? '')) {
                Log::warning('WhatsAppBot: Invalid webhook token', [
                    'ip' => $request->ip(),
                    'user_agent' => $request->userAgent()
                ]);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid webhook token'
                ], 401);
            }

            // SECURITY: Rate limiting per IP
            $cacheKey = 'whatsapp_webhook_' . $request->ip();
            $attempts = cache()->get($cacheKey, 0);
            if ($attempts > 60) { // Max 60 requests per minute
                Log::warning('WhatsAppBot: Rate limit exceeded', ['ip' => $request->ip()]);
                return response()->json(['status' => 'error', 'message' => 'Rate limit exceeded'], 429);
            }
            cache()->put($cacheKey, $attempts + 1, 60);

            // Get the raw payload from NearBuyMarts (includes all fields they send)
            $payload = $request->only([
                'sender', 'receiver', 'text', 'token', 
                'key_id', 'recnumber', 'recname', 'devtype', 
                'mentioned', 'appurl', 'nodeurl', 'rawdata', 'verifiedBizName',
                // Also support alternative field names for compatibility
                'from', 'phone', 'number', 'message', 'msgtext', 'body', 'timestamp'
            ]);

            // Log the full payload for debugging (first time setup)
            Log::info('WhatsAppBot: Raw webhook payload received', [
                'payload_keys' => array_keys($payload),
                'has_sender' => isset($payload['sender']),
                'has_text' => isset($payload['text']),
                'has_receiver' => isset($payload['receiver']),
            ]);

            // Extract sender and message from NearBuyMarts format
            // Primary: NearBuyMarts format (sender, text)
            // Fallback: Alternative formats for compatibility
            $sender = data_get($payload, 'sender') 
                   ?? data_get($payload, 'from') 
                   ?? data_get($payload, 'phone')
                   ?? data_get($payload, 'number');

            $message = data_get($payload, 'text')        // NearBuyMarts primary field
                    ?? data_get($payload, 'message') 
                    ?? data_get($payload, 'msgtext') 
                    ?? data_get($payload, 'body');

            $receiver = data_get($payload, 'receiver');  // Your bot number

            // SECURITY: Validate and sanitize inputs
            if (empty($sender) || !is_string($sender)) {
                Log::warning('WhatsAppBot: Invalid sender in webhook', [
                    'payload' => $payload
                ]);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Valid sender phone number is required'
                ], 400);
            }

            if (empty($message) || !is_string($message)) {
                Log::warning('WhatsAppBot: Invalid message in webhook', [
                    'sender' => substr($sender, 0, 8) . '***',
                    'payload_keys' => array_keys($payload)
                ]);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Valid message text is required'
                ], 400);
            }

            // SECURITY: Limit message length to prevent DoS
            if (strlen($message) > 4096) {
                Log::warning('WhatsAppBot: Message too long', ['length' => strlen($message)]);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Message too long'
                ], 400);
            }

            // Log incoming message (sanitized)
            Log::info('WhatsAppBot: Processing incoming message', [
                'correlation_id' => $correlation_id,
                'sender' => substr($sender, 0, 8) . '***',
                'receiver' => $receiver ? substr($receiver, 0, 8) . '***' : 'unknown',
                'message_preview' => substr($message, 0, 50),
                'message_length' => strlen($message),
                'ip' => $request->ip(),
            ]);

            // Deprecated token check removed - now mandatory above
            $verifyToken = config('whatsappbot.verify_webhook_token');
            if ($verifyToken) {
                $requestToken = $request->header('X-Webhook-Token') 
                             ?? $request->input('token');
                
                if ($requestToken !== $verifyToken) {
                    Log::warning('WhatsAppBot: Invalid webhook token', [
                        'expected' => $verifyToken,
                        'received' => $requestToken
                    ]);
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Invalid webhook token'
                    ], 401);
                }
            }

            // Process the message asynchronously (recommended for production)
            // For now, we'll process it synchronously
            $reply = $this->bot->processMessage($sender, $message, array_merge($payload, [
                'correlation_id' => $correlation_id
            ]));

            // Build meta routing hints for NearBuyMarts to deliver replies reliably
            // Prefer remoteJid from raw payload to derive address format
            $remoteJid = data_get($payload, 'rawdata.key.remoteJid')
                      ?? data_get($payload, 'key.remoteJid');

            $receiver_format = null;
            $lid_value = null;
            $addrToInspect = $remoteJid ?: $receiver;
            if (is_string($addrToInspect)) {
                if (str_ends_with($addrToInspect, '@lid')) {
                    $receiver_format = 'business_lid';
                    $lid_value = explode('@', $addrToInspect)[0] ?? null;
                } elseif (str_ends_with($addrToInspect, '@s.whatsapp.net')) {
                    $receiver_format = 'regular_whatsapp';
                } elseif (str_ends_with($addrToInspect, '@g.us')) {
                    $receiver_format = 'group';
                }
            }

            // Try to hint business phone for fallback routing (multi-tenant safe)
            $business_phone_hint = null;
            try {
                // Prefer resolving by recname or verifiedBizName from payload (maps to business name)
                $recname = data_get($payload, 'recname')
                        ?? data_get($payload, 'rawdata.verifiedBizName')
                        ?? data_get($payload, 'rawdata.pushName');

                if (!empty($recname)) {
                    $biz = \DB::table('business')->where('name', $recname)->first();
                    if ($biz && !empty($biz->whatsapp_phone)) {
                        $business_phone_hint = preg_replace('/\D/', '', $biz->whatsapp_phone);
                    }
                }

                // Fallback to current session business if available
                if ($business_phone_hint === null) {
                    $business_id = session()->get('user.business_id');
                    if ($business_id) {
                        $business = \DB::table('business')->where('id', $business_id)->first();
                        if ($business && !empty($business->whatsapp_phone)) {
                            $business_phone_hint = preg_replace('/\D/', '', $business->whatsapp_phone);
                        }
                    }
                }
            } catch (\Throwable $t) {
                // Non-fatal; proceed without phone hint
            }

            // Return HTTP 200 quickly so webhook provider knows we received it
            // Include the reply so external systems (like NearBuyMarts) can forward it
            return response()->json([
                'status' => 'ok',
                'message' => 'Message received and processed',
                'queued' => false,
                'reply_sent' => !empty($reply),
                'reply' => $reply ?? null,  // Include the actual reply text
                'meta' => [
                    'correlation_id' => $correlation_id,
                    'routing' => [
                        'type' => $receiver_format,
                        'lid' => $receiver_format === 'business_lid' ? $lid_value : null,
                        'phone_hint' => $business_phone_hint,
                    ],
                    'receiver_format' => $receiver_format,
                    'sender_format' => is_string($sender) && str_contains($sender, '@') ? 'jid' : 'phone'
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('WhatsAppBot: Error processing webhook', [
                'correlation_id' => isset($correlation_id) ? $correlation_id : null,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'payload' => $request->all()
            ]);

            // Return 200 to prevent webhook retries, but log the error
            return response()->json([
                'status' => 'error',
                'message' => 'Internal error processing message'
            ], 200);
        }
    }

    /**
     * Verify webhook endpoint (GET request)
     * Some webhook providers ping this to verify the endpoint is accessible
     */
    public function verifyWebhook(Request $request)
    {
        // Log webhook verification attempt
        Log::info('WhatsAppBot: Webhook verification request', [
            'ip' => $request->ip(),
            'query_params' => $request->query()
        ]);

        // Return success response
        return response()->json([
            'status' => 'ok',
            'message' => 'Webhook endpoint is active',
            'timestamp' => time(),
            'endpoint' => 'WhatsApp Bot Webhook'
        ], 200);
    }

    /**
     * Display settings page.
     * @return Renderable
     */
    public function settings()
    {
        $business_id = session()->get('user.business_id');
        $business = \DB::table('business')->where('id', $business_id)->first();
        
        return view('whatsappbot::settings', compact('business'));
    }

    /**
     * Update WhatsApp settings.
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateSettings(Request $request)
    {
        // SECURITY: Strict validation with regex patterns
        $request->validate([
            'whatsapp_phone' => ['nullable', 'string', 'max:20', 'regex:/^[+]?[0-9]{10,15}$/'],
            'whatsapp_enabled' => 'nullable|boolean',
            'auto_send_invoices' => 'nullable|boolean',
        ]);

        try {
            // SECURITY: Verify user has access to this business
            $business_id = session()->get('user.business_id');
            $user_id = auth()->id();
            
            // Verify business exists and user has access
            $business = \DB::table('business')
                ->where('id', $business_id)
                ->first();

            if (!$business) {
                Log::warning('WhatsAppBot: Business not found', [
                    'user_id' => $user_id,
                    'business_id' => $business_id
                ]);
                abort(404, 'Business not found.');
            }

            // Check if user owns the business (basic authorization)
            // Note: business_users table may not exist in all installations
            if (isset($business->owner_id) && $business->owner_id != $user_id) {
                // Additional check: Is user a superadmin?
                if (!auth()->user()->can('superadmin')) {
                    Log::warning('WhatsAppBot: Unauthorized settings update attempt', [
                        'user_id' => $user_id,
                        'business_id' => $business_id
                    ]);
                    abort(403, 'Unauthorized action.');
                }
            }
            
            // SECURITY: Use parameter binding to prevent SQL injection
            \DB::table('business')
                ->where('id', $business_id)
                ->update([
                    'whatsapp_phone' => $request->input('whatsapp_phone'),
                    'whatsapp_enabled' => (bool)$request->input('whatsapp_enabled', false),
                    'auto_send_invoices' => (bool)$request->input('auto_send_invoices', false),
                    'updated_at' => now(),
                ]);

            // Clear any cached business data
            cache()->forget('business_' . $business_id);

            return redirect()->route('whatsappbot.settings')
                           ->with('status', ['success' => true, 'msg' => __('lang_v1.success')]);

        } catch (\Exception $e) {
            Log::error('WhatsAppBot: Settings update failed', [
                'error' => $e->getMessage(),
                'user_id' => auth()->id()
            ]);
            
            return redirect()->route('whatsappbot.settings')
                           ->with('status', ['success' => false, 'msg' => __('lang_v1.something_went_wrong')]);
        }
    }

    /**
     * Display message logs.
     * @return Renderable
     */
    public function logs()
    {
        // For now, return a basic view
        // In production, you might fetch logs from database or log files
        return view('whatsappbot::logs');
    }

    /**
     * Display test messages page.
     * @return Renderable
     */
    public function testMessages()
    {
        $business_id = session()->get('user.business_id');
        $business = \DB::table('business')->where('id', $business_id)->first();
        
        return view('whatsappbot::test-messages', compact('business'));
    }

    /**
     * Send a test WhatsApp message (for admin testing)
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function sendTest(Request $request)
    {
        $request->validate([
            'phone' => 'required|string',
            'message' => 'required|string',
        ]);

        try {
            $phone = $request->input('phone');
            $message = $request->input('message');

            $result = $this->bot->sendTestMessage($phone, $message);

            // Validate the API response
            if ($result === null) {
                Log::error('WhatsAppBot: Test message failed - no response from API', [
                    'phone' => $phone
                ]);
                
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to send message: No response from WhatsApp API'
                ], 500);
            }
            
            // Parse the response to check for success
            $responseData = json_decode($result, true);
            
            if (isset($responseData['success']) && $responseData['success'] === false) {
                $errorMessage = $responseData['message'] ?? 'Unknown error from WhatsApp API';
                
                Log::error('WhatsAppBot: Test message failed - API returned error', [
                    'phone' => $phone,
                    'api_error' => $errorMessage
                ]);
                
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to send message: ' . $errorMessage
                ], 500);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Test message sent successfully',
                'result' => $result
            ]);

        } catch (\Exception $e) {
            Log::error('WhatsAppBot: Test message failed', [
                'error' => $e->getMessage(),
                'phone' => $request->input('phone')
            ]);

            return response()->json([
                'status' => 'error',
                'message' => 'Failed to send test message: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show webhook tester page
     */
    public function webhookTester()
    {
        return view('whatsappbot::webhook-tester');
    }

    /**
     * Test webhook by simulating an incoming message
     */
    public function testWebhook(Request $request)
    {
        try {
            $sender = $request->input('sender');
            $message = $request->input('message');

            if (empty($sender) || empty($message)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Sender and message are required'
                ], 400);
            }

            Log::info('WhatsAppBot: Manual webhook test', [
                'sender' => substr($sender, 0, 8) . '***',
                'message' => substr($message, 0, 30)
            ]);

            // Process the message through the bot
            $reply = $this->bot->processMessage($sender, $message, [
                'test' => true,
                'timestamp' => time()
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Webhook test successful',
                'reply' => $reply
            ]);

        } catch (\Exception $e) {
            Log::error('WhatsAppBot: Webhook test failed', [
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => 'error',
                'message' => 'Webhook test failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get recent webhook logs
     */
    public function webhookLogs()
    {
        try {
            $logFile = storage_path('logs/laravel.log');
            
            if (!file_exists($logFile)) {
                return response()->json([
                    'logs' => [],
                    'message' => 'Log file not found'
                ]);
            }

            // Read last 200 lines of log file
            $lines = [];
            $file = new \SplFileObject($logFile, 'r');
            $file->seek(PHP_INT_MAX);
            $lastLine = $file->key();
            $startLine = max(0, $lastLine - 200);
            
            $file->seek($startLine);
            while (!$file->eof()) {
                $line = $file->current();
                if (stripos($line, 'WhatsAppBot') !== false || stripos($line, 'webhook') !== false) {
                    $lines[] = $line;
                }
                $file->next();
            }

            // Parse log lines
            $logs = [];
            foreach (array_slice($lines, -20) as $line) {
                if (preg_match('/\[(.*?)\].*?(INFO|ERROR|WARNING|DEBUG):\s*(.+)/', $line, $matches)) {
                    $logs[] = [
                        'timestamp' => $matches[1],
                        'level' => strtolower($matches[2]),
                        'message' => trim($matches[3]),
                        'context' => null
                    ];
                }
            }

            return response()->json([
                'logs' => array_reverse($logs)
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'logs' => [],
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
