# WhatsAppBot Module - Quick Installation Reference

## 🚀 Installation Format: Superadmin Standard

### Quick Reference

**Module Name**: `whatsappbot`  
**Display Name**: `WhatsAppBot`  
**Version**: `1.0.0`  
**Format**: Superadmin (Centralized License Validation)

---

## Installation URLs

```
Install:    GET  /whatsappbot/install
            POST /whatsappbot/install

Update:     GET  /whatsappbot/install/update
            POST /whatsappbot/install/update

Uninstall:  GET  /whatsappbot/install/uninstall
```

---

## Controller Methods

| Method | Purpose | HTTP | Redirects To |
|--------|---------|------|-------------|
| `index()` | Show install form | GET | - |
| `install()` | Execute install | POST | ModulesController |
| `update()` | Show update form | GET | - |
| `updateModule()` | Execute update | POST | ModulesController |
| `uninstall()` | Execute uninstall | GET | ModulesController |

---

## System Model Methods Used

```php
// Check installation
System::getProperty('whatsappbot_version')

// Install
System::addProperty('whatsappbot_version', '1.0.0')

// Update
System::setProperty('whatsappbot_version', '1.1.0')

// Uninstall
System::removeProperty('whatsappbot_version')
```

---

## Installation Process

1. **User navigates to**: `/whatsappbot/install`
2. **System checks**: If already installed → abort(404)
3. **Shows**: Shared form at `resources/views/install/install-module.blade.php`
4. **User submits**: License code, username, email
5. **System runs**: `Artisan::call('module:migrate', ['module' => 'WhatsAppBot'])`
6. **System stores**: Version in system table via `System::addProperty()`
7. **Redirects to**: Central modules controller

---

## Key Files

```
✓ Config/config.php (has module_version)
✓ Http/Controllers/InstallController.php (Superadmin format)
✓ Routes/web.php (simplified routes)
✗ Resources/views/install/ (deleted - uses shared view)
```

---

## Required Variables for Shared View

```php
$action_url = action([InstallController::class, 'install']);
$intruction_type = 'cc'; // or 'iu' for update
$action_type = 'install'; // or 'update'
$module_display_name = 'WhatsAppBot';
```

---

## Version Management

**Config file**: `Modules/WhatsAppBot/Config/config.php`
```php
'module_version' => '1.0.0'
```

**Module file**: `Modules/WhatsAppBot/module.json`
```json
"version": "1.0.0"
```

**System table**: Key = `whatsappbot_version`, Value = `1.0.0`

---

## Permissions

All installation routes require:
```php
auth()->user()->can('superadmin')
```

Non-superadmin users get **403 Forbidden**.

---

## Response Format

```php
// Success
$output = [
    'success' => 1,
    'msg' => 'WhatsAppBot module installed successfully!'
];

// Error
$output = [
    'success' => 0,
    'msg' => $e->getMessage()
];

// Redirect
return redirect()
    ->action([ModulesController::class, 'index'])
    ->with('status', $output);
```

---

## Migrations

```php
Artisan::call('module:migrate', [
    'module' => 'WhatsAppBot',
    '--force' => true
]);
```

Creates:
- `whatsapp_bot_logs` table
- Adds phone columns to `business` table

---

## Error Handling

```php
try {
    DB::beginTransaction();
    // ... operations ...
    DB::commit();
} catch (Exception $e) {
    DB::rollBack();
    \Log::emergency('File: ' . $e->getFile() . ' Line: ' . $e->getLine() . ' Message: ' . $e->getMessage());
}
```

---

## Version Comparison

```php
use Composer\Semver\Comparator;

$is_updated = Comparator::greaterThan($this->appVersion, $version_installed) 
    ? false  // needs update
    : true;  // already updated
```

---

## Installation Settings

```php
private function installSettings()
{
    $business_id = request()->session()->get('user.business_id');
    $user_id = request()->session()->get('user.id');
}

// Called at start of each installation method
ini_set('max_execution_time', 0);
ini_set('memory_limit', '512M');
$this->installSettings();
```

---

## Differences from Old Format

| Feature | Old (AdvancedRestaurant) | New (Superadmin) |
|---------|-------------------------|------------------|
| Storage | Direct DB queries | System model |
| View | Custom view | Shared view |
| Response | JSON | Redirect |
| License | No check | Has validation |
| Routes | Nested prefix | Flat structure |

---

## Status Check

After installation, check system table:

```sql
SELECT * FROM system WHERE `key` = 'whatsappbot_version';
```

Expected result:
```
key: whatsappbot_version
value: 1.0.0
```

---

## Troubleshooting

**404 on install page**: Module already installed  
**404 on update page**: Module not installed or already updated  
**403 Forbidden**: User is not superadmin  
**Migration errors**: Check database permissions  
**Rollback executed**: Check Laravel log for exact error  

---

## Complete! ✅

WhatsAppBot module now uses the standard Superadmin installation format with centralized license validation and consistent UX across all JebbMS modules.
