# WhatsAppBot Module - Migration Error Fix

## Issue
During installation, the following error occurred:
```
SQLSTATE[42S22]: Column not found: 1054 Unknown column 'email' in 'business' 
(SQL: alter table `business` add `phone` varchar(32) null after `email`, 
add `owner_phone` varchar(32) null after `phone`, 
add `whatsapp_enabled` tinyint(1) not null default '0' after `owner_phone`)
```

## Root Cause
The migration was trying to add columns after the `email` column, but the `business` table doesn't have an `email` column. Additionally, there was already an existing `whatsapp_phone` column in the business table from a previous migration.

## Solution
Updated the WhatsAppBot module to use existing database columns instead of creating new ones:

### 1. **Fixed Migration File**
**File**: `Modules/WhatsAppBot/Database/Migrations/2025_11_06_031341_add_phone_columns_to_business_table.php`

**BEFORE**:
```php
// Tried to add phone, owner_phone, whatsapp_enabled after 'email' column
$table->string('phone', 32)->nullable()->after('email');
$table->string('owner_phone', 32)->nullable()->after('phone');
$table->boolean('whatsapp_enabled')->default(false)->after('owner_phone');
```

**AFTER**:
```php
// Only add whatsapp_enabled flag since whatsapp_phone already exists
if (!Schema::hasColumn('business', 'whatsapp_enabled')) {
    $table->boolean('whatsapp_enabled')->default(false);
}
```

### 2. **Updated Service to Use Existing Column**
**File**: `Modules/WhatsAppBot/Services/WhatsAppBotService.php`

**BEFORE**:
```php
$business = DB::table('business')
    ->where('phone', $phone)
    ->orWhere('owner_phone', $phone)
    ->first();
```

**AFTER**:
```php
$business = DB::table('business')
    ->where('whatsapp_phone', $phone)
    ->first();
```

### 3. **Updated Controller to Use Existing Column**
**File**: `Modules/WhatsAppBot/Http/Controllers/WhatsAppBotController.php`

**BEFORE**:
```php
$business_phone = $business->phone ?? null;
```

**AFTER**:
```php
$business_phone = $business->whatsapp_phone ?? null;
```

### 4. **Updated Install Controller Rollback**
**File**: `Modules/WhatsAppBot/Http/Controllers/InstallController.php`

**BEFORE**:
```php
// Tried to drop phone and owner_phone columns
if (Schema::hasColumn('business', 'phone')) {
    $table->dropColumn('phone');
}
if (Schema::hasColumn('business', 'owner_phone')) {
    $table->dropColumn('owner_phone');
}
```

**AFTER**:
```php
// Only handle whatsapp_enabled column (our addition)
if (Schema::hasColumn('business', 'whatsapp_enabled')) {
    $table->dropColumn('whatsapp_enabled');
}
```

### 5. **Updated Configuration Comments**
**File**: `Modules/WhatsAppBot/Config/config.php`

**BEFORE**:
```php
// Set to 'business_table' to use phone column in business table
// Set to 'custom' to use separate whatsapp_business_phones table
```

**AFTER**:
```php
// Uses existing 'whatsapp_phone' column in business table
// This column is created by existing migration: 2025_10_29_202944_add_whatsapp_fields_to_users_and_businesses_tables.php
```

## Database Schema Integration

### Existing WhatsApp Fields (Already Available)
From migration `2025_10_29_202944_add_whatsapp_fields_to_users_and_businesses_tables.php`:

**Business Table**:
- `whatsapp_phone` (string, 20) - Phone number for WhatsApp
- `whatsapp_settings` (json) - WhatsApp configuration settings

**Users Table**:
- `whatsapp_phone` (string, 20) - User's WhatsApp phone
- `whatsapp_notifications_enabled` (boolean) - Notification preference
- `whatsapp_consent_at` (timestamp) - When user gave consent
- `whatsapp_opt_out_at` (timestamp) - When user opted out

### New WhatsAppBot Fields (Added by this module)
**Business Table**:
- `whatsapp_enabled` (boolean, default: false) - Enable/disable WhatsApp bot for business

**WhatsApp Bot Logs Table**:
- `id` (increments)
- `business_id` (integer) - Foreign key to business
- `phone` (string, 32) - Sender's phone number
- `message_type` (enum) - incoming/outgoing
- `message_content` (text) - Message content
- `response_content` (text) - Bot response
- `created_at` / `updated_at` (timestamps)

## How Phone Resolution Works Now

1. **WhatsApp message received** with sender phone number
2. **System normalizes** phone number (handles +234, 0, etc.)
3. **System queries**: `SELECT * FROM business WHERE whatsapp_phone = ?`
4. **If business found**: Process message for that business context
5. **If business not found**: Send generic "business not found" message

## Business Phone Setup

To enable WhatsApp bot for a business:

1. **Set WhatsApp Phone**:
   ```sql
   UPDATE business 
   SET whatsapp_phone = '+2348012345678' 
   WHERE id = 1;
   ```

2. **Enable WhatsApp Bot**:
   ```sql
   UPDATE business 
   SET whatsapp_enabled = 1 
   WHERE id = 1;
   ```

3. **Configure WhatsApp Settings** (optional):
   ```sql
   UPDATE business 
   SET whatsapp_settings = '{"auto_reply": true, "business_hours": "9:00-17:00"}' 
   WHERE id = 1;
   ```

## Admin Dashboard Changes

The admin dashboard now shows:
- **Business Phone**: Displays `whatsapp_phone` instead of `phone`
- **Status**: Shows if business has WhatsApp phone configured
- **Test Message**: Uses `whatsapp_phone` for testing

## Migration Dependencies

This module now depends on:
1. **Core migration**: `2025_10_29_202944_add_whatsapp_fields_to_users_and_businesses_tables.php`
   - Creates `whatsapp_phone` column in business table
   - Creates `whatsapp_settings` column in business table

2. **WhatsAppBot migrations**:
   - `2025_11_06_031341_add_phone_columns_to_business_table.php` (updated)
   - `2025_11_06_031342_create_whatsapp_bot_logs_table.php` (unchanged)

## Benefits of This Approach

✅ **No Column Conflicts** - Uses existing whatsapp_phone column  
✅ **Consistent Data** - All WhatsApp features use same phone field  
✅ **Cleaner Schema** - No duplicate phone columns  
✅ **Easier Maintenance** - Single source of truth for WhatsApp phone  
✅ **Better Integration** - Works with existing WhatsApp features  

## Testing Instructions

### 1. Clean Installation
```bash
# Remove any previous failed installation
php artisan module:migrate-rollback WhatsAppBot

# Run fresh installation
php artisan module:migrate WhatsAppBot --force
```

### 2. Verify Database Schema
```sql
-- Check business table has required columns
DESCRIBE business;
-- Should show: whatsapp_phone, whatsapp_settings, whatsapp_enabled

-- Check whatsapp_bot_logs table created
DESCRIBE whatsapp_bot_logs;
-- Should show: id, business_id, phone, message_type, etc.
```

### 3. Test Business Phone Resolution
```php
// In tinker or test script
$business = DB::table('business')->where('whatsapp_phone', '+2348012345678')->first();
// Should return business record if phone is set
```

### 4. Test WhatsApp Bot Installation
1. Navigate to `/whatsappbot/install`
2. Complete installation with license details
3. Check system table has `whatsappbot_version = 1.0.0`
4. Verify no migration errors in Laravel log

## Status: ✅ FIXED

The migration error has been resolved. The WhatsAppBot module now:
- Uses existing `whatsapp_phone` column from business table
- Only adds `whatsapp_enabled` flag (no positioning conflicts)
- Integrates properly with existing WhatsApp infrastructure
- Follows DRY principle (Don't Repeat Yourself) for phone storage

Installation should now complete successfully without column errors.