<?php

namespace Modules\WhatsAppBot\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Transaction;
use App\Business;

class WhatsAppInvoiceLog extends Model
{
    use HasFactory;

    protected $table = 'whatsapp_invoice_logs';

    protected $fillable = [
        'business_id',
        'transaction_id',
        'invoice_no',
        'customer_phone',
        'customer_name',
        'message',
        'pdf_filename',
        'status',
        'api_response',
        'error_message',
        'sent_at',
        'retry_count',
    ];

    protected $casts = [
        'sent_at' => 'datetime',
        'api_response' => 'array',
        'retry_count' => 'integer',
    ];

    /**
     * Get the business that owns the log
     */
    public function business()
    {
        return $this->belongsTo(Business::class);
    }

    /**
     * Get the transaction that owns the log
     */
    public function transaction()
    {
        return $this->belongsTo(Transaction::class);
    }

    /**
     * Scope for successful sends
     */
    public function scopeSent($query)
    {
        return $query->where('status', 'sent');
    }

    /**
     * Scope for failed sends
     */
    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    /**
     * Scope for pending sends
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope for recent logs
     */
    public function scopeRecent($query, $days = 7)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    /**
     * Mark as sent
     */
    public function markAsSent($apiResponse = null)
    {
        $this->update([
            'status' => 'sent',
            'sent_at' => now(),
            'api_response' => $apiResponse,
        ]);
    }

    /**
     * Mark as failed
     */
    public function markAsFailed($errorMessage = null, $apiResponse = null)
    {
        $this->update([
            'status' => 'failed',
            'error_message' => $errorMessage,
            'api_response' => $apiResponse,
            'retry_count' => $this->retry_count + 1,
        ]);
    }

    /**
     * Check if can retry
     */
    public function canRetry($maxRetries = 3)
    {
        return $this->retry_count < $maxRetries && $this->status === 'failed';
    }
}