# WhatsApp Bot Module for JebbMS Ultimate POS

## Overview

The WhatsApp Bot module enables business owners to query their business data directly via WhatsApp using natural language. The module integrates with the NearBuyMarts WhatsApp API to provide real-time business intelligence.

## Features

### 📊 Sales Queries
- Today's sales
- Yesterday's sales
- This week's sales
- This month's sales
- Transaction counts

### 💰 Profit Queries
- Today's profit
- Yesterday's profit
- This week's profit
- This month's profit
- Breakdown by sales and purchases

### 📈 Reports
- Full daily business reports
- Top selling products
- Best-selling items analysis

### 🔒 Security
- Phone number to business mapping
- Multi-tenant data isolation
- Optional webhook token verification
- Comprehensive logging

## Installation

### 1. Prerequisites
- JebbMS Ultimate POS (Laravel-based)
- NearBuyMarts WhatsApp API account
- PHP 7.3+
- MySQL/MariaDB

### 2. Module Setup

The module is already installed in your `Modules/WhatsAppBot` directory with the following structure:

```
Modules/WhatsAppBot/
├── Config/
│   └── config.php
├── Database/
│   └── Migrations/
│       ├── 2025_11_06_031341_add_phone_columns_to_business_table.php
│       └── 2025_11_06_031342_create_whatsapp_bot_logs_table.php
├── Http/
│   └── Controllers/
│       └── WhatsAppBotController.php
├── Providers/
│   ├── WhatsAppBotServiceProvider.php
│   └── RouteServiceProvider.php
├── Resources/
│   ├── lang/
│   │   └── en/
│   │       └── lang.php
│   └── views/
│       └── index.blade.php
├── Routes/
│   ├── api.php
│   └── web.php
├── Services/
│   └── WhatsAppBotService.php
├── composer.json
└── module.json
```

### 3. Environment Configuration

Add the following to your `.env` file:

```env
# NearBuyMarts WhatsApp API Configuration
WHATSAPP_SEND_NODEURL=https://api.nearbuymarts.com/send
WHATSAPP_SEND_TOKEN=UzgsBoKOeg6Y9w1JeIda
WHATSAPP_MEDIA_URL=https://app.nearbuymarts.com/public/app-assets/images/logo/logo.png

# Optional: Webhook security token
WHATSAPP_WEBHOOK_VERIFY_TOKEN=your-secure-token-here

# Optional: Phone mapping method (business_table or custom)
WHATSAPP_PHONE_MAPPING=business_table

# Optional: Logging
WHATSAPP_LOG_INCOMING=true
WHATSAPP_LOG_OUTGOING=true
```

### 4. Run Database Migrations

```bash
cd /Users/engrabiolaadebowale/Downloads/UpdateJEB6.8.1/jebbms-oct-2025
php artisan migrate
```

This will:
- Add `phone`, `owner_phone`, and `whatsapp_enabled` columns to the `business` table
- Create `whatsapp_bot_logs` table for message logging

### 5. Register Business Phone Numbers

Update each business record with their WhatsApp phone number:

```sql
UPDATE business 
SET phone = '+2348012345678', 
    whatsapp_enabled = 1 
WHERE id = 1;
```

Or via the admin panel:
1. Go to Business Settings
2. Add WhatsApp phone number (with country code)
3. Enable WhatsApp bot

### 6. Configure Webhook in NearBuyMarts

1. Log in to your NearBuyMarts dashboard
2. Navigate to API Settings → Webhooks
3. Add webhook URL:
   ```
   https://yourdomain.com/api/webhook/whatsapp
   ```
4. Set webhook method to `POST`
5. Add any required authentication tokens

### 7. Access the Admin Panel

Navigate to: `https://yourdomain.com/whatsappbot`

From here you can:
- View configuration status
- Copy webhook URL
- Send test messages
- View available commands
- Access setup instructions

## Usage

### For Business Owners

Send WhatsApp messages to your registered business number with queries like:

**Sales Queries:**
- "What's today's sales?"
- "Show yesterday's sales"
- "This week's sales"
- "This month's sales"

**Profit Queries:**
- "What's today's profit?"
- "Yesterday's profit"
- "This week's profit"
- "This month's profit"

**Reports:**
- "Send report"
- "Full report"
- "Top products"
- "Best selling items"

**Help:**
- "Help"
- "Commands"
- "What can you do?"

### Response Format

The bot responds with formatted messages containing:
- 📊 Emojis for visual clarity
- ₦ Currency formatting (Naira)
- 📅 Date/time stamps
- 📈 Transaction counts
- 💰 Profit breakdowns

Example response:
```
🛒 Today's Sales - My Store

Date: 06 Nov 2024
Total Sales: ₦125,450.00
Transactions: 23

Generated at: 03:14 PM
```

## API Endpoints

### Webhook Endpoint (Incoming Messages)
- **URL:** `POST /api/webhook/whatsapp`
- **Purpose:** Receives incoming WhatsApp messages from NearBuyMarts
- **Payload:**
  ```json
  {
    "sender": "+2348012345678",
    "message": "What's today's sales?",
    "timestamp": "2024-11-06 15:30:00"
  }
  ```

### Test Message Endpoint
- **URL:** `POST /api/whatsapp/send-test`
- **Purpose:** Send test messages from admin panel
- **Auth:** Required
- **Payload:**
  ```json
  {
    "phone": "+2348012345678",
    "message": "Test message"
  }
  ```

### Web Dashboard
- **URL:** `GET /whatsappbot`
- **Purpose:** Admin configuration and testing interface

## Security Features

### Multi-Tenant Isolation
All queries are automatically scoped to the business associated with the sender's phone number:

```php
DB::table('transactions')
    ->where('business_id', $businessId)
    ->where('type', 'sell')
    ->sum('final_total');
```

### Phone Number Verification
Only registered phone numbers can query business data. Unregistered numbers receive:

```
⚠️ Business Not Found

We couldn't find a business linked to this phone number.

Please:
1. Register your number in the POS system
2. Contact support for assistance
```

### Optional Webhook Token Verification
Enable webhook token verification in config:

```php
'verify_webhook_token' => env('WHATSAPP_WEBHOOK_VERIFY_TOKEN', null),
```

### Comprehensive Logging
All incoming and outgoing messages are logged to:
- Laravel log files
- `whatsapp_bot_logs` database table

## Database Schema

### Business Table (Updated)
```sql
ALTER TABLE business 
ADD COLUMN phone VARCHAR(32) NULL,
ADD COLUMN owner_phone VARCHAR(32) NULL,
ADD COLUMN whatsapp_enabled BOOLEAN DEFAULT 0;
```

### WhatsApp Bot Logs Table
```sql
CREATE TABLE whatsapp_bot_logs (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    business_id INT UNSIGNED,
    phone VARCHAR(32),
    direction ENUM('incoming', 'outgoing'),
    message TEXT,
    payload JSON,
    status VARCHAR(50),
    response TEXT,
    processed_at TIMESTAMP,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (business_id) REFERENCES business(id) ON DELETE CASCADE
);
```

## Customization

### Adding New Query Types

Edit `/Modules/WhatsAppBot/Services/WhatsAppBotService.php`:

```php
// Add in processMessage() method
if ($this->matchesQuery($lower, ['custom', 'query'])) {
    $text = $this->getCustomQueryText($business->id, $business->name);
    $this->sendWhatsAppMessage($sender, $text);
    return $text;
}

// Add custom method
protected function getCustomQueryText($businessId, $businessName)
{
    // Your custom logic here
    $data = DB::table('your_table')
        ->where('business_id', $businessId)
        ->get();
    
    return "Your formatted response";
}
```

### Changing Response Format

Modify the text generation methods in `WhatsAppBotService.php`:

```php
protected function getTodaySalesText($businessId, $businessName)
{
    $sales = $this->getTodaySalesAmount($businessId);
    
    // Customize this format
    $text = "🛒 *Today's Sales*\n";
    $text .= "Amount: ₦" . number_format($sales, 2);
    
    return $text;
}
```

### Adding NLP/AI Integration

Replace keyword matching with AI:

```php
public function processMessage($sender, $message, $payload = [])
{
    // Use OpenAI, Google DialogFlow, etc.
    $intent = $this->detectIntent($message);
    
    switch ($intent) {
        case 'sales_query':
            return $this->handleSalesQuery($message, $business);
        case 'profit_query':
            return $this->handleProfitQuery($message, $business);
        // ...
    }
}
```

## Troubleshooting

### Messages Not Received

1. **Check webhook URL:** Ensure it's publicly accessible via HTTPS
2. **Verify NearBuyMarts configuration:** Check API dashboard
3. **Check logs:** `storage/logs/laravel.log`
4. **Test endpoint manually:**
   ```bash
   curl -X POST https://yourdomain.com/api/webhook/whatsapp \
     -H "Content-Type: application/json" \
     -d '{"sender":"+234801234567","message":"test"}'
   ```

### No Business Found Error

1. **Verify phone format:** Must match exactly (with + and country code)
2. **Check database:** 
   ```sql
   SELECT id, name, phone FROM business WHERE phone LIKE '%8012345678%';
   ```
3. **Update phone number:**
   ```sql
   UPDATE business SET phone = '+2348012345678' WHERE id = 1;
   ```

### Messages Not Sending

1. **Verify API token:** Check `WHATSAPP_SEND_TOKEN` in .env
2. **Check API endpoint:** Ensure NearBuyMarts API is accessible
3. **Review logs:** Look for cURL errors
4. **Test API directly:**
   ```bash
   curl -X POST https://api.nearbuymarts.com/send \
     -d "receiver=+234801234567" \
     -d "msgtext=Test" \
     -d "token=YOUR_TOKEN"
   ```

## Performance Optimization

### Enable Query Caching

```php
protected function getTodaySalesAmount($businessId)
{
    return Cache::remember("today_sales_{$businessId}", 300, function() use ($businessId) {
        return DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereDate('transaction_date', Carbon::today())
            ->where('type', 'sell')
            ->sum('final_total');
    });
}
```

### Queue Message Processing

```php
public function handleIncoming(Request $request)
{
    dispatch(new ProcessWhatsAppMessage($sender, $message, $payload));
    
    return response()->json(['status' => 'queued']);
}
```

## Support

For issues or questions:
- **Email:** support@jebbms.com
- **Documentation:** https://docs.jebbms.com
- **GitHub Issues:** (if applicable)

## License

This module is part of the JebbMS Ultimate POS system and follows the same licensing terms.

## Changelog

### Version 1.0.0 (2024-11-06)
- Initial release
- Sales and profit queries
- Daily/weekly/monthly reports
- Top products analysis
- NearBuyMarts API integration
- Multi-tenant support
- Admin dashboard
- Comprehensive logging
