@extends('layouts.app')
@section('title', __('whatsappbot::lang.webhook_tester'))

@section('content')
<section class="content-header">
    <h1>@lang('whatsappbot::lang.webhook_tester')
        <small>Test webhook configuration</small>
    </h1>
</section>

<section class="content">
    <div class="row">
        <div class="col-md-8">
            <div class="box box-primary">
                <div class="box-header with-border">
                    <h3 class="box-title">
                        <i class="fa fa-plug"></i> Webhook Configuration
                    </h3>
                </div>
                <div class="box-body">
                    <div class="alert alert-info">
                        <i class="fa fa-info-circle"></i>
                        <strong>Your Webhook URL:</strong><br>
                        <code id="webhook-url">{{ url('api/webhook/whatsapp') }}</code>
                        <button class="btn btn-xs btn-default" onclick="copyToClipboard('webhook-url')">
                            <i class="fa fa-copy"></i> Copy
                        </button>
                    </div>

                    <div class="alert alert-warning">
                        <i class="fa fa-key"></i>
                        <strong>Webhook Token:</strong><br>
                        <code id="webhook-token">{{ config('whatsappbot.verify_webhook_token') }}</code>
                        <button class="btn btn-xs btn-default" onclick="copyToClipboard('webhook-token')">
                            <i class="fa fa-copy"></i> Copy
                        </button>
                    </div>

                    <div class="alert alert-success">
                        <i class="fa fa-phone"></i>
                        <strong>Central Bot Number:</strong><br>
                        <code>{{ config('whatsappbot.central_bot_number') }}</code>
                    </div>

                    <hr>

                    <h4><i class="fa fa-cog"></i> NearBuyMarts Configuration</h4>
                    <p>Configure these settings in your NearBuyMarts dashboard:</p>
                    
                    <ol>
                        <li>Login to <a href="https://nearbuymarts.com" target="_blank">NearBuyMarts Dashboard</a></li>
                        <li>Go to <strong>Settings → Webhooks</strong> or <strong>API Configuration</strong></li>
                        <li>Add the following configuration:</li>
                    </ol>

                    <table class="table table-bordered">
                        <tr>
                            <th width="30%">Setting</th>
                            <th>Value</th>
                        </tr>
                        <tr>
                            <td>Webhook URL</td>
                            <td><code>{{ url('api/webhook/whatsapp') }}</code></td>
                        </tr>
                        <tr>
                            <td>Authentication Method</td>
                            <td>Header-based (Custom Header)</td>
                        </tr>
                        <tr>
                            <td>Header Name</td>
                            <td><code>X-Webhook-Token</code></td>
                        </tr>
                        <tr>
                            <td>Header Value</td>
                            <td><code>{{ config('whatsappbot.verify_webhook_token') }}</code></td>
                        </tr>
                        <tr>
                            <td>Alternative (Query String)</td>
                            <td><code>{{ url('api/webhook/whatsapp') }}?token={{ config('whatsappbot.verify_webhook_token') }}</code></td>
                        </tr>
                        <tr>
                            <td>Bot Number</td>
                            <td><code>{{ config('whatsappbot.central_bot_number') }}</code></td>
                        </tr>
                    </table>
                </div>
            </div>

            <div class="box box-success">
                <div class="box-header with-border">
                    <h3 class="box-title">
                        <i class="fa fa-check-circle"></i> Test Webhook
                    </h3>
                </div>
                <div class="box-body">
                    <p>Test if your webhook is receiving and processing messages correctly.</p>
                    
                    <form id="webhook-test-form">
                        <div class="form-group">
                            <label>Sender Phone Number</label>
                            <input type="text" class="form-control" id="test-sender" 
                                   value="{{ auth()->user()->contact->mobile ?? '2348136666556' }}" 
                                   placeholder="2348136666556">
                        </div>

                        <div class="form-group">
                            <label>Test Message</label>
                            <select class="form-control" id="test-command" onchange="updateTestMessage()">
                                <option value="">-- Select a command --</option>
                                <option value="Help">Help - Show available commands</option>
                                <option value="What's today's sales?">Sales Today</option>
                                <option value="Show yesterday's sales">Sales Yesterday</option>
                                <option value="This week's sales">Sales This Week</option>
                                <option value="What's today's profit?">Profit Today</option>
                                <option value="Send report">Generate Report</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label>Custom Message (or use dropdown above)</label>
                            <textarea class="form-control" id="test-message" rows="3" placeholder="Type your test message here">Help</textarea>
                        </div>

                        <button type="submit" class="btn btn-success">
                            <i class="fa fa-paper-plane"></i> Send Test Webhook
                        </button>
                    </form>

                    <div id="test-result" style="margin-top: 20px;"></div>
                </div>
            </div>
        </div>

        <div class="col-md-4">
            <div class="box box-info">
                <div class="box-header with-border">
                    <h3 class="box-title">
                        <i class="fa fa-question-circle"></i> How It Works
                    </h3>
                </div>
                <div class="box-body">
                    <ol style="padding-left: 20px;">
                        <li><strong>User sends message</strong> to your WhatsApp bot number</li>
                        <li><strong>NearBuyMarts receives</strong> the message</li>
                        <li><strong>NearBuyMarts sends webhook</strong> to your URL with token</li>
                        <li><strong>Your system validates</strong> the token</li>
                        <li><strong>Command is processed</strong> and response generated</li>
                        <li><strong>Response is sent back</strong> via NearBuyMarts API</li>
                    </ol>

                    <hr>

                    <h4><i class="fa fa-exclamation-triangle"></i> Troubleshooting</h4>
                    
                    <div class="small">
                        <strong>No response from bot?</strong>
                        <ul>
                            <li>Verify webhook URL is configured in NearBuyMarts</li>
                            <li>Check authentication token matches</li>
                            <li>Ensure bot number is linked to webhook</li>
                            <li>Check logs below</li>
                        </ul>

                        <strong>401 Unauthorized?</strong>
                        <ul>
                            <li>Token mismatch - verify token in NearBuyMarts</li>
                            <li>Missing X-Webhook-Token header</li>
                        </ul>

                        <strong>503 Service Unavailable?</strong>
                        <ul>
                            <li>Webhook token not configured in .env</li>
                            <li>Run: <code>php artisan config:clear</code></li>
                        </ul>
                    </div>
                </div>
            </div>

            <div class="box box-warning">
                <div class="box-header with-border">
                    <h3 class="box-title">
                        <i class="fa fa-list"></i> Recent Webhook Logs
                    </h3>
                    <div class="box-tools">
                        <button class="btn btn-xs btn-default" onclick="refreshLogs()">
                            <i class="fa fa-refresh"></i> Refresh
                        </button>
                    </div>
                </div>
                <div class="box-body" id="webhook-logs" style="max-height: 400px; overflow-y: auto;">
                    <p class="text-muted"><i>Loading logs...</i></p>
                </div>
            </div>
        </div>
    </div>
</section>
@endsection

@section('javascript')
<script>
function updateTestMessage() {
    var select = document.getElementById('test-command');
    var textarea = document.getElementById('test-message');
    if (select.value) {
        textarea.value = select.value;
    }
}

function copyToClipboard(elementId) {
    var element = document.getElementById(elementId);
    var text = element.textContent;
    
    if (navigator.clipboard) {
        navigator.clipboard.writeText(text).then(function() {
            toastr.success('Copied to clipboard!');
        });
    } else {
        // Fallback for older browsers
        var textArea = document.createElement("textarea");
        textArea.value = text;
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
        toastr.success('Copied to clipboard!');
    }
}

function refreshLogs() {
    $('#webhook-logs').html('<p class="text-muted"><i class="fa fa-spinner fa-spin"></i> Loading logs...</p>');
    
    $.ajax({
        url: '{{ route("whatsappbot.webhook-logs") }}',
        method: 'GET',
        success: function(response) {
            if (response.logs && response.logs.length > 0) {
                var html = '<div class="small">';
                response.logs.forEach(function(log) {
                    var icon = log.level === 'error' ? 'fa-times-circle text-danger' : 
                              log.level === 'warning' ? 'fa-exclamation-triangle text-warning' : 
                              'fa-check-circle text-success';
                    
                    html += '<div style="margin-bottom: 10px; border-left: 3px solid #ddd; padding-left: 10px;">';
                    html += '<small class="text-muted">' + log.timestamp + '</small><br>';
                    html += '<i class="fa ' + icon + '"></i> ';
                    html += '<strong>' + log.message + '</strong><br>';
                    if (log.context) {
                        html += '<code style="font-size: 10px;">' + JSON.stringify(log.context).substring(0, 100) + '</code>';
                    }
                    html += '</div>';
                });
                html += '</div>';
                $('#webhook-logs').html(html);
            } else {
                $('#webhook-logs').html('<p class="text-muted">No webhook logs found. Send a message to your bot to see logs here.</p>');
            }
        },
        error: function() {
            $('#webhook-logs').html('<p class="text-danger"><i class="fa fa-exclamation-circle"></i> Error loading logs</p>');
        }
    });
}

$(document).ready(function() {
    // Load logs on page load
    refreshLogs();

    // Test webhook form submission
    $('#webhook-test-form').on('submit', function(e) {
        e.preventDefault();
        
        var sender = $('#test-sender').val().trim();
        var message = $('#test-message').val().trim();
        
        if (!sender || !message) {
            toastr.error('Please fill in all fields');
            return;
        }

        $('#test-result').html('<div class="alert alert-info"><i class="fa fa-spinner fa-spin"></i> Sending test webhook...</div>');

        $.ajax({
            url: '{{ route("whatsappbot.test-webhook") }}',
            method: 'POST',
            data: {
                _token: '{{ csrf_token() }}',
                sender: sender,
                message: message
            },
            success: function(response) {
                var alertClass = response.status === 'success' ? 'alert-success' : 'alert-danger';
                var icon = response.status === 'success' ? 'fa-check-circle' : 'fa-times-circle';
                
                var html = '<div class="alert ' + alertClass + '">';
                html += '<i class="fa ' + icon + '"></i> <strong>' + response.message + '</strong><br>';
                if (response.reply) {
                    html += '<hr style="margin: 10px 0;">';
                    html += '<strong>Bot Reply:</strong><br>';
                    html += '<em>' + response.reply + '</em>';
                }
                if (response.error) {
                    html += '<br><small>' + response.error + '</small>';
                }
                html += '</div>';
                
                $('#test-result').html(html);
                
                // Refresh logs after test
                setTimeout(refreshLogs, 1000);
            },
            error: function(xhr) {
                var message = 'Error: ';
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    message += xhr.responseJSON.message;
                } else {
                    message += 'HTTP ' + xhr.status + ' - ' + xhr.statusText;
                }
                
                $('#test-result').html('<div class="alert alert-danger"><i class="fa fa-times-circle"></i> ' + message + '</div>');
            }
        });
    });

    // Auto-refresh logs every 30 seconds
    setInterval(refreshLogs, 30000);
});
</script>
@endsection
