<?php

namespace Modules\WhatsAppBot\Services;

use App\Services\InternationalPhoneFormatter;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Modules\WhatsAppBot\Models\WhatsAppInvoiceLog;

class WhatsAppBotService
{
    /**
     * @var string NearBuyMarts API endpoint
     */
    protected $sendNodeUrl;

    /**
     * @var string API token
     */
    protected $token;

    /**
     * @var string Optional media URL for branding
     */
    protected $mediaurl;

    /**
     * Initialize service with configuration
     */
    public function __construct()
    {
        $this->sendNodeUrl = config('whatsappbot.nodeurl', 'https://api.nearbuymarts.com/send');
        $this->token = config('whatsappbot.token', 'UzgsBoKOeg6Y9w1JeIda');
        $this->mediaurl = config('whatsappbot.mediaurl');
    }

    /**
     * Main dispatcher for incoming WhatsApp messages
     *
     * @param string $sender Phone number of sender
     * @param string $message Message text
     * @param array $payload Full webhook payload
     * @return string|null Reply text sent to user
     */
    public function processMessage($sender, $message, $payload = [])
    {
        // Normalize phone number
        $sender = $this->normalizePhone($sender);
        $message = trim((string) ($message ?? ''));
        $lower = Str::lower($message);

        // Log the incoming message
        Log::info('WhatsAppBot: Processing message', [
            'sender' => $sender,
            'message' => $message
        ]);

        // Check if central bot mode is enabled
        $mappingMethod = config('whatsappbot.phone_mapping_method', 'central_bot');
        
        if ($mappingMethod === 'central_bot') {
            // Central bot mode: identify business from message content or user registration
            $business = $this->resolveBusinessFromMessage($sender, $message);
        } else {
            // Legacy mode: identify business from sender phone number
            $business = $this->resolveBusinessFromPhone($sender);
        }
        
        if (!$business) {
            if ($mappingMethod === 'central_bot') {
                $replyText = $this->getCentralBotWelcomeText();
            } else {
                $replyText = "⚠️ Business Not Found\n\n";
                $replyText .= "We couldn't find a business linked to this phone number.\n\n";
                $replyText .= "Please:\n";
                $replyText .= "1. Register your number in the POS system\n";
                $replyText .= "2. Contact support for assistance\n\n";
                $replyText .= "Support: support@jebbms.com";
            }
            
            $this->sendWhatsAppMessage($sender, $replyText);
            return $replyText;
        }

        // Process different query types based on message content
        // Using simple keyword matching - can be replaced with NLP/LLM for more sophisticated parsing

        // TODAY'S PROFIT QUERIES
        if ($this->matchesQuery($lower, ['today', 'profit'])) {
            $text = $this->getTodayProfitText($business->id, $business->name);
            $this->sendWhatsAppMessage($sender, $text);
            return $text;
        }

        // TODAY'S SALES QUERIES
        if ($this->matchesQuery($lower, ['today', 'sales'])) {
            $text = $this->getTodaySalesText($business->id, $business->name);
            $this->sendWhatsAppMessage($sender, $text);
            return $text;
        }

        // YESTERDAY'S QUERIES
        if ($this->matchesQuery($lower, ['yesterday', 'sales'])) {
            $text = $this->getYesterdaySalesText($business->id, $business->name);
            $this->sendWhatsAppMessage($sender, $text);
            return $text;
        }

        if ($this->matchesQuery($lower, ['yesterday', 'profit'])) {
            $text = $this->getYesterdayProfitText($business->id, $business->name);
            $this->sendWhatsAppMessage($sender, $text);
            return $text;
        }

        // THIS WEEK QUERIES
        if ($this->matchesQuery($lower, ['week', 'sales'])) {
            $text = $this->getWeekSalesText($business->id, $business->name);
            $this->sendWhatsAppMessage($sender, $text);
            return $text;
        }

        if ($this->matchesQuery($lower, ['week', 'profit'])) {
            $text = $this->getWeekProfitText($business->id, $business->name);
            $this->sendWhatsAppMessage($sender, $text);
            return $text;
        }

        // THIS MONTH QUERIES
        if ($this->matchesQuery($lower, ['month', 'sales'])) {
            $text = $this->getMonthSalesText($business->id, $business->name);
            $this->sendWhatsAppMessage($sender, $text);
            return $text;
        }

        if ($this->matchesQuery($lower, ['month', 'profit'])) {
            $text = $this->getMonthProfitText($business->id, $business->name);
            $this->sendWhatsAppMessage($sender, $text);
            return $text;
        }

        // FULL REPORT
        if ($this->matchesQuery($lower, ['report', 'send']) || 
            $this->matchesQuery($lower, ['report', 'submit']) ||
            $this->matchesQuery($lower, ['full', 'report'])) {
            $report = $this->buildTodayReport($business->id, $business->name);
            $this->sendWhatsAppMessage($sender, $report);
            return $report;
        }

        // TOP PRODUCTS
        if ($this->matchesQuery($lower, ['top', 'product']) || 
            $this->matchesQuery($lower, ['best', 'selling'])) {
            $text = $this->getTopProductsText($business->id, $business->name);
            $this->sendWhatsAppMessage($sender, $text);
            return $text;
        }

        // HELP/COMMANDS
        if ($this->matchesQuery($lower, ['help']) || 
            $this->matchesQuery($lower, ['commands']) ||
            $this->matchesQuery($lower, ['what', 'can'])) {
            $text = $this->getHelpText();
            $this->sendWhatsAppMessage($sender, $text);
            return $text;
        }

        // Fallback: Send welcome message with available commands
        $text = $this->getWelcomeText($business->name);
        $this->sendWhatsAppMessage($sender, $text);
        return $text;
    }

    /**
     * Check if message matches query keywords
     *
     * @param string $message Lowercase message
     * @param array $keywords Keywords to match
     * @return bool
     */
    protected function matchesQuery($message, $keywords)
    {
        foreach ($keywords as $keyword) {
            if (!Str::contains($message, $keyword)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Get welcome text with available commands
     *
     * @param string $businessName
     * @return string
     */
    protected function getWelcomeText($businessName)
    {
        $text = "👋 *Welcome to {$businessName} Bot*\n\n";
        $text .= "I can help you with:\n\n";
        $text .= "📊 *Sales & Profit:*\n";
        $text .= "• Today's sales\n";
        $text .= "• Today's profit\n";
        $text .= "• Yesterday's sales\n";
        $text .= "• This week's sales\n";
        $text .= "• This month's profit\n\n";
        $text .= "📈 *Reports:*\n";
        $text .= "• Send report\n";
        $text .= "• Top products\n\n";
        $text .= "Reply *'help'* to see examples\n";
        $text .= "Reply *'commands'* for full list";

        return $text;
    }

    /**
     * Get help text with command examples
     *
     * @return string
     */
    protected function getHelpText()
    {
        $text = "📋 *Available Commands:*\n\n";
        $text .= "*Sales Queries:*\n";
        $text .= "• What's today's sales?\n";
        $text .= "• Show yesterday's sales\n";
        $text .= "• This week's sales\n";
        $text .= "• This month's sales\n\n";
        $text .= "*Profit Queries:*\n";
        $text .= "• What's today's profit?\n";
        $text .= "• Yesterday's profit\n";
        $text .= "• This week's profit\n";
        $text .= "• This month's profit\n\n";
        $text .= "*Reports:*\n";
        $text .= "• Send report\n";
        $text .= "• Full report\n";
        $text .= "• Top products\n";
        $text .= "• Best selling items\n\n";
        $text .= "💡 Tip: Just ask naturally, I'll understand!";

        return $text;
    }

    /**
     * Normalize phone number to consistent format
     * Uses InternationalPhoneFormatter for automatic country detection
     *
     * @param string $phone
     * @param int|null $businessId
     * @return string
     */
    protected function normalizePhone($phone, $businessId = null)
    {
        return InternationalPhoneFormatter::formatInternational($phone, $businessId);
    }

    /**
     * Resolve business from phone number
     *
     * @param string $phone
     * @return object|null Business record
     */
    protected function resolveBusinessFromPhone($phone)
    {
        try {
            // SECURITY: Use parameter binding to prevent SQL injection
            $business = DB::table('business')
                ->where('whatsapp_phone', '=', $phone)
                ->where('whatsapp_enabled', '=', true)
                ->first();

            if ($business) {
                Log::info('WhatsAppBot: Business resolved', [
                    'business_id' => $business->id,
                    'business_name' => $business->name
                ]);
            }

            return $business;

        } catch (\Exception $e) {
            Log::error('WhatsAppBot: Error resolving business', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Resolve business from message content (Central Bot Mode)
     *
     * @param string $senderPhone Customer/user phone number
     * @param string $message Message content
     * @return object|null Business record
     */
    protected function resolveBusinessFromMessage($senderPhone, $message)
    {
        try {
            // First, check if this user has a saved business preference
            $savedBusiness = $this->getUserBusinessPreference($senderPhone);
            if ($savedBusiness) {
                return $savedBusiness;
            }

            // Check if message contains business identifier
            $businessFromMessage = $this->extractBusinessFromMessage($message);
            if ($businessFromMessage) {
                // Save this preference for future messages
                $this->saveUserBusinessPreference($senderPhone, $businessFromMessage->id);
                return $businessFromMessage;
            }

            // If no business found, prompt user to identify their business
            return null;

        } catch (\Exception $e) {
            Log::error('WhatsAppBot: Error resolving business from message', [
                'sender' => $senderPhone,
                'message' => $message,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Get user's saved business preference
     *
     * @param string $phone
     * @return object|null
     */
    protected function getUserBusinessPreference($phone)
    {
        try {
            // SECURITY: Use parameter binding
            $preference = DB::table('whatsapp_user_businesses')
                ->where('user_phone', '=', $phone)
                ->where('is_active', '=', true)
                ->first();

            if ($preference) {
                return DB::table('business')
                    ->where('id', '=', $preference->business_id)
                    ->where('whatsapp_enabled', '=', true)
                    ->first();
            }

            return null;
        } catch (\Exception $e) {
            // Table might not exist yet
            return null;
        }
    }

    /**
     * Save user business preference
     *
     * @param string $phone
     * @param int $businessId
     */
    protected function saveUserBusinessPreference($phone, $businessId)
    {
        try {
            // SECURITY: Validate inputs
            if (!is_numeric($businessId) || $businessId <= 0) {
                Log::warning('WhatsAppBot: Invalid business ID');
                return;
            }

            // Deactivate existing preferences (use parameter binding)
            DB::table('whatsapp_user_businesses')
                ->where('user_phone', '=', $phone)
                ->update(['is_active' => false, 'updated_at' => now()]);

            // Insert new preference with parameter binding
            DB::table('whatsapp_user_businesses')->insert([
                'user_phone' => $phone,
                'business_id' => (int)$businessId,
                'is_active' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

        } catch (\Exception $e) {
            Log::warning('WhatsAppBot: Could not save user business preference', [
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Extract business from message content
     *
     * @param string $message
     * @return object|null
     */
    protected function extractBusinessFromMessage($message)
    {
        $lower = Str::lower($message);

        // Check for business name in message
        $businesses = DB::table('business')
            ->where('whatsapp_enabled', true)
            ->get();

        foreach ($businesses as $business) {
            $businessName = Str::lower($business->name);
            if (Str::contains($lower, $businessName)) {
                return $business;
            }

            // Check for business code/abbreviation if available
            if (!empty($business->code) && Str::contains($lower, Str::lower($business->code))) {
                return $business;
            }
        }

        return null;
    }

    /**
     * Get central bot welcome text with business selection
     *
     * @return string
     */
    protected function getCentralBotWelcomeText()
    {
        $text = "🤖 *Welcome to JEBBMS Business Bot!*\n\n";
        $text .= "To get started, please specify your business:\n\n";
        
        // Get available businesses
        $businesses = DB::table('business')
            ->where('whatsapp_enabled', true)
            ->take(10)
            ->get();

        if ($businesses->count() > 0) {
            $text .= "*Available Businesses:*\n";
            foreach ($businesses as $business) {
                $text .= "• {$business->name}\n";
            }
            $text .= "\n📝 *How to use:*\n";
            $text .= "Type: \"[Business Name] today sales\"\n";
            $text .= "Example: \"ABC Store today sales\"\n\n";
        }

        $text .= "*Available Commands:*\n";
        $text .= "• [Business] today sales\n";
        $text .= "• [Business] today profit\n";
        $text .= "• [Business] send report\n";
        $text .= "• [Business] top products\n\n";
        $text .= "💡 Once you use a business name, I'll remember it for future messages!";

        return $text;
    }

    // =====================================================================
    // SALES QUERY METHODS
    // =====================================================================

    /**
     * Get today's total sales amount
     *
     * @param int $businessId
     * @return float
     */
    protected function getTodaySalesAmount($businessId)
    {
        $today = Carbon::today();
        
        return DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereDate('transaction_date', $today)
            ->where('type', 'sell')
            ->whereNull('deleted_at')
            ->sum('final_total');
    }

    /**
     * Get today's sales formatted text
     *
     * @param int $businessId
     * @param string $businessName
     * @return string
     */
    protected function getTodaySalesText($businessId, $businessName)
    {
        $sales = $this->getTodaySalesAmount($businessId);
        $count = $this->getTodayTransactionCount($businessId);
        
        $text = "🛒 *Today's Sales - {$businessName}*\n\n";
        $text .= "Date: " . Carbon::today()->format('d M Y') . "\n";
        $text .= "Total Sales: ₦" . number_format($sales, 2) . "\n";
        $text .= "Transactions: {$count}\n\n";
        $text .= "Generated at: " . Carbon::now()->format('h:i A');

        return $text;
    }

    /**
     * Get yesterday's sales text
     *
     * @param int $businessId
     * @param string $businessName
     * @return string
     */
    protected function getYesterdaySalesText($businessId, $businessName)
    {
        $yesterday = Carbon::yesterday();
        
        $sales = DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereDate('transaction_date', $yesterday)
            ->where('type', 'sell')
            ->whereNull('deleted_at')
            ->sum('final_total');

        $count = DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereDate('transaction_date', $yesterday)
            ->where('type', 'sell')
            ->whereNull('deleted_at')
            ->count();
        
        $text = "🛒 *Yesterday's Sales - {$businessName}*\n\n";
        $text .= "Date: " . $yesterday->format('d M Y') . "\n";
        $text .= "Total Sales: ₦" . number_format($sales, 2) . "\n";
        $text .= "Transactions: {$count}\n\n";
        $text .= "Generated at: " . Carbon::now()->format('h:i A');

        return $text;
    }

    /**
     * Get this week's sales text
     *
     * @param int $businessId
     * @param string $businessName
     * @return string
     */
    protected function getWeekSalesText($businessId, $businessName)
    {
        $weekStart = Carbon::now()->startOfWeek();
        $weekEnd = Carbon::now()->endOfWeek();
        
        $sales = DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereBetween('transaction_date', [$weekStart, $weekEnd])
            ->where('type', 'sell')
            ->whereNull('deleted_at')
            ->sum('final_total');

        $count = DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereBetween('transaction_date', [$weekStart, $weekEnd])
            ->where('type', 'sell')
            ->whereNull('deleted_at')
            ->count();
        
        $text = "🛒 *This Week's Sales - {$businessName}*\n\n";
        $text .= "Period: " . $weekStart->format('d M') . " - " . $weekEnd->format('d M Y') . "\n";
        $text .= "Total Sales: ₦" . number_format($sales, 2) . "\n";
        $text .= "Transactions: {$count}\n\n";
        $text .= "Generated at: " . Carbon::now()->format('h:i A');

        return $text;
    }

    /**
     * Get this month's sales text
     *
     * @param int $businessId
     * @param string $businessName
     * @return string
     */
    protected function getMonthSalesText($businessId, $businessName)
    {
        $monthStart = Carbon::now()->startOfMonth();
        $monthEnd = Carbon::now()->endOfMonth();
        
        $sales = DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereBetween('transaction_date', [$monthStart, $monthEnd])
            ->where('type', 'sell')
            ->whereNull('deleted_at')
            ->sum('final_total');

        $count = DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereBetween('transaction_date', [$monthStart, $monthEnd])
            ->where('type', 'sell')
            ->whereNull('deleted_at')
            ->count();
        
        $text = "🛒 *This Month's Sales - {$businessName}*\n\n";
        $text .= "Month: " . $monthStart->format('F Y') . "\n";
        $text .= "Total Sales: ₦" . number_format($sales, 2) . "\n";
        $text .= "Transactions: {$count}\n\n";
        $text .= "Generated at: " . Carbon::now()->format('h:i A');

        return $text;
    }

    // =====================================================================
    // PROFIT QUERY METHODS
    // =====================================================================

    /**
     * Get today's purchase amount
     *
     * @param int $businessId
     * @return float
     */
    protected function getTodayPurchaseAmount($businessId)
    {
        $today = Carbon::today();
        
        return DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereDate('transaction_date', $today)
            ->where('type', 'purchase')
            ->whereNull('deleted_at')
            ->sum('final_total');
    }

    /**
     * Get today's profit text
     *
     * @param int $businessId
     * @param string $businessName
     * @return string
     */
    protected function getTodayProfitText($businessId, $businessName)
    {
        $sales = $this->getTodaySalesAmount($businessId);
        $purchases = $this->getTodayPurchaseAmount($businessId);
        $profit = $sales - $purchases;
        
        $text = "💰 *Today's Profit - {$businessName}*\n\n";
        $text .= "Date: " . Carbon::today()->format('d M Y') . "\n";
        $text .= "Sales: ₦" . number_format($sales, 2) . "\n";
        $text .= "Purchases: ₦" . number_format($purchases, 2) . "\n";
        $text .= "Profit: ₦" . number_format($profit, 2) . "\n\n";
        $text .= "Generated at: " . Carbon::now()->format('h:i A');

        return $text;
    }

    /**
     * Get yesterday's profit text
     *
     * @param int $businessId
     * @param string $businessName
     * @return string
     */
    protected function getYesterdayProfitText($businessId, $businessName)
    {
        $yesterday = Carbon::yesterday();
        
        $sales = DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereDate('transaction_date', $yesterday)
            ->where('type', 'sell')
            ->whereNull('deleted_at')
            ->sum('final_total');

        $purchases = DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereDate('transaction_date', $yesterday)
            ->where('type', 'purchase')
            ->whereNull('deleted_at')
            ->sum('final_total');

        $profit = $sales - $purchases;
        
        $text = "💰 *Yesterday's Profit - {$businessName}*\n\n";
        $text .= "Date: " . $yesterday->format('d M Y') . "\n";
        $text .= "Sales: ₦" . number_format($sales, 2) . "\n";
        $text .= "Purchases: ₦" . number_format($purchases, 2) . "\n";
        $text .= "Profit: ₦" . number_format($profit, 2) . "\n\n";
        $text .= "Generated at: " . Carbon::now()->format('h:i A');

        return $text;
    }

    /**
     * Get this week's profit text
     *
     * @param int $businessId
     * @param string $businessName
     * @return string
     */
    protected function getWeekProfitText($businessId, $businessName)
    {
        $weekStart = Carbon::now()->startOfWeek();
        $weekEnd = Carbon::now()->endOfWeek();
        
        $sales = DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereBetween('transaction_date', [$weekStart, $weekEnd])
            ->where('type', 'sell')
            ->whereNull('deleted_at')
            ->sum('final_total');

        $purchases = DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereBetween('transaction_date', [$weekStart, $weekEnd])
            ->where('type', 'purchase')
            ->whereNull('deleted_at')
            ->sum('final_total');

        $profit = $sales - $purchases;
        
        $text = "💰 *This Week's Profit - {$businessName}*\n\n";
        $text .= "Period: " . $weekStart->format('d M') . " - " . $weekEnd->format('d M Y') . "\n";
        $text .= "Sales: ₦" . number_format($sales, 2) . "\n";
        $text .= "Purchases: ₦" . number_format($purchases, 2) . "\n";
        $text .= "Profit: ₦" . number_format($profit, 2) . "\n\n";
        $text .= "Generated at: " . Carbon::now()->format('h:i A');

        return $text;
    }

    /**
     * Get this month's profit text
     *
     * @param int $businessId
     * @param string $businessName
     * @return string
     */
    protected function getMonthProfitText($businessId, $businessName)
    {
        $monthStart = Carbon::now()->startOfMonth();
        $monthEnd = Carbon::now()->endOfMonth();
        
        $sales = DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereBetween('transaction_date', [$monthStart, $monthEnd])
            ->where('type', 'sell')
            ->whereNull('deleted_at')
            ->sum('final_total');

        $purchases = DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereBetween('transaction_date', [$monthStart, $monthEnd])
            ->where('type', 'purchase')
            ->whereNull('deleted_at')
            ->sum('final_total');

        $profit = $sales - $purchases;
        
        $text = "💰 *This Month's Profit - {$businessName}*\n\n";
        $text .= "Month: " . $monthStart->format('F Y') . "\n";
        $text .= "Sales: ₦" . number_format($sales, 2) . "\n";
        $text .= "Purchases: ₦" . number_format($purchases, 2) . "\n";
        $text .= "Profit: ₦" . number_format($profit, 2) . "\n\n";
        $text .= "Generated at: " . Carbon::now()->format('h:i A');

        return $text;
    }

    // =====================================================================
    // REPORT METHODS
    // =====================================================================

    /**
     * Get transaction count for today
     *
     * @param int $businessId
     * @return int
     */
    protected function getTodayTransactionCount($businessId)
    {
        $today = Carbon::today();
        
        return DB::table('transactions')
            ->where('business_id', $businessId)
            ->whereDate('transaction_date', $today)
            ->where('type', 'sell')
            ->whereNull('deleted_at')
            ->count();
    }

    /**
     * Build comprehensive daily report
     *
     * @param int $businessId
     * @param string $businessName
     * @return string
     */
    protected function buildTodayReport($businessId, $businessName)
    {
        $sales = $this->getTodaySalesAmount($businessId);
        $purchases = $this->getTodayPurchaseAmount($businessId);
        $profit = $sales - $purchases;
        $count = $this->getTodayTransactionCount($businessId);

        $text = "📊 *Daily Business Report*\n";
        $text .= "*{$businessName}*\n";
        $text .= "━━━━━━━━━━━━━━━━━━\n\n";
        $text .= "📅 Date: " . Carbon::today()->format('d F Y') . "\n\n";
        $text .= "💵 *Financial Summary:*\n";
        $text .= "Total Sales: ₦" . number_format($sales, 2) . "\n";
        $text .= "Total Purchases: ₦" . number_format($purchases, 2) . "\n";
        $text .= "Net Profit: ₦" . number_format($profit, 2) . "\n\n";
        $text .= "📈 *Transaction Summary:*\n";
        $text .= "Total Transactions: {$count}\n";
        
        if ($count > 0) {
            $avgTransaction = $sales / $count;
            $text .= "Average Sale: ₦" . number_format($avgTransaction, 2) . "\n\n";
        } else {
            $text .= "Average Sale: ₦0.00\n\n";
        }

        $text .= "━━━━━━━━━━━━━━━━━━\n";
        $text .= "Generated: " . Carbon::now()->format('d M Y h:i A') . "\n";
        $text .= "\nThank you for using JebbMS! 🎉";

        return $text;
    }

    /**
     * Get top selling products today
     *
     * @param int $businessId
     * @param string $businessName
     * @return string
     */
    protected function getTopProductsText($businessId, $businessName)
    {
        $today = Carbon::today();
        
        // Get top 5 products sold today
        $topProducts = DB::table('transaction_sell_lines')
            ->join('transactions', 'transaction_sell_lines.transaction_id', '=', 'transactions.id')
            ->join('products', 'transaction_sell_lines.product_id', '=', 'products.id')
            ->where('transactions.business_id', $businessId)
            ->whereDate('transactions.transaction_date', $today)
            ->where('transactions.type', 'sell')
            ->whereNull('transactions.deleted_at')
            ->select(
                'products.name',
                DB::raw('SUM(transaction_sell_lines.quantity) as total_quantity'),
                DB::raw('SUM(transaction_sell_lines.unit_price_inc_tax * transaction_sell_lines.quantity) as total_amount')
            )
            ->groupBy('products.id', 'products.name')
            ->orderBy('total_quantity', 'desc')
            ->limit(5)
            ->get();

        $text = "🏆 *Top Selling Products - {$businessName}*\n\n";
        $text .= "Date: " . Carbon::today()->format('d M Y') . "\n\n";

        if ($topProducts->isEmpty()) {
            $text .= "No sales recorded today.\n\n";
        } else {
            $rank = 1;
            foreach ($topProducts as $product) {
                $emoji = $rank == 1 ? '🥇' : ($rank == 2 ? '🥈' : ($rank == 3 ? '🥉' : '▪️'));
                $text .= "{$emoji} *{$product->name}*\n";
                $text .= "   Qty: {$product->total_quantity} | ₦" . number_format($product->total_amount, 2) . "\n\n";
                $rank++;
            }
        }

        $text .= "Generated at: " . Carbon::now()->format('h:i A');

        return $text;
    }

    // =====================================================================
    // WHATSAPP MESSAGING METHODS
    // =====================================================================

    /**
     * Send WhatsApp message via NearBuyMarts API
     *
     * @param string $receiver Phone number
     * @param string $message Message text
     * @param string|null $mediaUrl Optional media URL
     * @return string API response
     */
    protected function sendWhatsAppMessage($receiver, $message, $mediaUrl = null)
    {
        try {
            // SECURITY: Validate receiver phone number
            if (!preg_match('/^[+]?[0-9]{10,15}$/', $receiver)) {
                Log::error('WhatsAppBot: Invalid receiver phone format', [
                    'receiver' => substr($receiver, 0, 20)
                ]);
                return null;
            }

            // SECURITY: Validate message length
            if (strlen($message) > 4096) {
                Log::error('WhatsAppBot: Message too long', [
                    'length' => strlen($message)
                ]);
                return null;
            }

            // SECURITY: Validate API endpoint (prevent SSRF)
            $allowedHosts = ['api.nearbuymarts.com'];
            $parsedUrl = parse_url($this->sendNodeUrl);
            if (!isset($parsedUrl['host']) || !in_array($parsedUrl['host'], $allowedHosts)) {
                Log::critical('WhatsAppBot: Invalid API endpoint detected', [
                    'url' => $this->sendNodeUrl
                ]);
                return null;
            }

            // Prepare data for NearBuyMarts API
            $data = [
                'receiver' => $receiver,
                'msgtext'  => $message,
                'token'    => $this->token,
            ];

            // SECURITY: Validate media URL if provided (prevent SSRF)
            if ($mediaUrl) {
                $parsedMedia = parse_url($mediaUrl);
                if (!isset($parsedMedia['scheme']) || !in_array($parsedMedia['scheme'], ['https'])) {
                    Log::warning('WhatsAppBot: Invalid media URL scheme', [
                        'url' => substr($mediaUrl, 0, 100)
                    ]);
                } elseif (isset($parsedMedia['host']) && !in_array($parsedMedia['host'], ['127.0.0.1', 'localhost'])) {
                    $data['mediaurl'] = $mediaUrl;
                }
            } elseif ($this->mediaurl) {
                $parsedMedia = parse_url($this->mediaurl);
                if (isset($parsedMedia['scheme']) && $parsedMedia['scheme'] === 'https') {
                    $data['mediaurl'] = $this->mediaurl;
                }
            }

            // Log outgoing message (sanitized - DO NOT log full message content)
            if (config('whatsappbot.log_outgoing_messages', true)) {
                Log::info('WhatsAppBot: Sending message', [
                    'receiver' => substr($receiver, 0, 15) . '***',
                    'message_length' => strlen($message),
                    'has_media' => isset($data['mediaurl'])
                ]);
            }

            // Initialize cURL with secure settings
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/x-www-form-urlencoded']);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
            curl_setopt($ch, CURLOPT_URL, $this->sendNodeUrl);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
            
            // SECURITY: ENABLE SSL verification (critical for production)
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            
            // SECURITY: Disable redirects to prevent SSRF
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);
            curl_setopt($ch, CURLOPT_MAXREDIRS, 0);

            // Execute request
            $response = curl_exec($ch);
            $error = curl_error($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($error) {
                Log::error('WhatsAppBot: cURL error sending message', [
                    'error' => $error,
                    'http_code' => $httpCode
                ]);
                return null;
            }

            // SECURITY: Validate response is not too large (prevent memory exhaustion)
            if (strlen($response ?? '') > 10240) {
                Log::warning('WhatsAppBot: Response too large', [
                    'size' => strlen($response)
                ]);
                $response = substr($response, 0, 10240);
            }

            // Log sanitized response
            $logData = [
                'receiver' => substr($receiver, 0, 10) . '***',
                'http_code' => $httpCode,
                'response_length' => strlen($response ?? '')
            ];

            // If error response, log the actual response for debugging
            if ($httpCode !== 200 && $httpCode !== 201) {
                $logData['api_response'] = substr($response, 0, 500);
                Log::warning('WhatsAppBot: API error response', $logData);
            } else {
                Log::info('WhatsAppBot: Message sent successfully', $logData);
            }

            return $response;

        } catch (\Exception $e) {
            Log::error('WhatsAppBot: Exception sending message', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Send test message (for admin panel)
     *
     * @param string $phone
     * @param string $message
     * @return string
     */
    public function sendTestMessage($phone, $message)
    {
        $phone = $this->normalizePhone($phone);
        return $this->sendWhatsAppMessage($phone, $message);
    }

    // =====================================================================
    // INVOICE METHODS
    // =====================================================================

    /**
     * Send invoice PDF via WhatsApp to customer
     *
     * @param object $transaction Transaction record
     * @param string $customerPhone Customer phone number
     * @param string $pdfPath Path to the generated PDF file
     * @return string|null API response
     */
    public function sendInvoicePDF($transaction, $customerPhone, $pdfPath)
    {
        try {
            $customerPhone = $this->normalizePhone($customerPhone);
            
            // Get business details
            $business = DB::table('business')->where('id', $transaction->business_id)->first();
            if (!$business) {
                Log::error('WhatsAppBot: Business not found for invoice', ['transaction_id' => $transaction->id]);
                return null;
            }

            // Build invoice message
            $message = $this->buildInvoiceMessage($transaction, $business);

            // Upload PDF to a publicly accessible URL first
            $pdfUrl = $this->uploadInvoicePDF($pdfPath, $transaction->invoice_no);
            
            if (!$pdfUrl) {
                Log::warning('WhatsAppBot: Could not upload PDF, sending text only', [
                    'transaction_id' => $transaction->id,
                    'invoice_no' => $transaction->invoice_no
                ]);
                return $this->sendWhatsAppMessage($customerPhone, $message);
            }

            // Send message with PDF attachment
            $response = $this->sendWhatsAppMessage($customerPhone, $message, $pdfUrl);

            // Get PDF filename for logging
            $pdfFilename = basename($pdfUrl);

            // Log invoice send
            $this->logInvoiceSend($transaction, $customerPhone, $response, $message, $pdfFilename);

            return $response;

        } catch (\Exception $e) {
            Log::error('WhatsAppBot: Error sending invoice PDF', [
                'transaction_id' => $transaction->id,
                'customer_phone' => $customerPhone,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Build invoice message text
     *
     * @param object $transaction
     * @param object $business
     * @return string
     */
    protected function buildInvoiceMessage($transaction, $business)
    {
        $message = "🧾 *Invoice from {$business->name}*\n\n";
        $message .= "Invoice No: *{$transaction->invoice_no}*\n";
        $message .= "Date: " . \Carbon\Carbon::parse($transaction->transaction_date)->format('d M Y, h:i A') . "\n";
        $message .= "Amount: ₦" . number_format($transaction->final_total, 2) . "\n\n";
        
        // Add payment status
        if ($transaction->payment_status == 'paid') {
            $message .= "✅ Status: *PAID*\n\n";
        } else {
            $message .= "⏳ Status: *PENDING*\n\n";
        }

        $message .= "📄 Please find your invoice attached below.\n\n";
        $message .= "Thank you for your business! 🙏\n\n";
        
        // Add business contact info if available
        if ($business->mobile) {
            $message .= "📞 Contact: {$business->mobile}\n";
        }
        if ($business->email) {
            $message .= "📧 Email: {$business->email}\n";
        }

        $message .= "\n*{$business->name}* - Your trusted partner";

        return $message;
    }

    /**
     * Upload invoice PDF to public storage and return URL
     *
     * @param string $pdfPath Local PDF file path
     * @param string $invoiceNo Invoice number for filename
     * @return string|null Public URL to PDF
     */
    protected function uploadInvoicePDF($pdfPath, $invoiceNo)
    {
        try {
            // SECURITY: Validate invoice number to prevent path traversal
            if (!preg_match('/^[a-zA-Z0-9_-]+$/', $invoiceNo)) {
                Log::error('WhatsAppBot: Invalid invoice number format', [
                    'invoice_no' => substr($invoiceNo, 0, 50)
                ]);
                return null;
            }

            // SECURITY: Validate PDF path is within allowed directory
            $realPath = realpath($pdfPath);
            $allowedBasePath = realpath(storage_path('app'));
            
            if (!$realPath || !$allowedBasePath || strpos($realPath, $allowedBasePath) !== 0) {
                Log::error('WhatsAppBot: PDF path outside allowed directory', [
                    'path' => $pdfPath
                ]);
                return null;
            }

            if (!file_exists($pdfPath)) {
                Log::error('WhatsAppBot: PDF file not found', ['path' => $pdfPath]);
                return null;
            }

            // SECURITY: Verify file is actually a PDF
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($finfo, $pdfPath);
            finfo_close($finfo);
            
            if ($mimeType !== 'application/pdf') {
                Log::error('WhatsAppBot: File is not a valid PDF', [
                    'mime_type' => $mimeType
                ]);
                return null;
            }

            // SECURITY: Check file size (max 10MB)
            $fileSize = filesize($pdfPath);
            if ($fileSize > 10 * 1024 * 1024) {
                Log::error('WhatsAppBot: PDF file too large', [
                    'size' => $fileSize
                ]);
                return null;
            }

            // Create a public storage directory for invoices
            $publicDir = public_path('uploads/invoices/whatsapp');
            if (!is_dir($publicDir)) {
                mkdir($publicDir, 0755, true);
            }

            // Generate unique filename with sanitized invoice number
            $safeInvoiceNo = preg_replace('/[^a-zA-Z0-9_-]/', '', $invoiceNo);
            $filename = 'invoice_' . $safeInvoiceNo . '_' . bin2hex(random_bytes(8)) . '.pdf';
            $publicPath = $publicDir . '/' . $filename;

            // SECURITY: Verify destination path is within public directory
            $realPublicPath = realpath($publicDir);
            if (!$realPublicPath || strpos(realpath(dirname($publicPath)), $realPublicPath) !== 0) {
                Log::error('WhatsAppBot: Invalid destination path');
                return null;
            }

            // Copy PDF to public directory
            if (!copy($pdfPath, $publicPath)) {
                Log::error('WhatsAppBot: Failed to copy PDF to public directory');
                return null;
            }

            // SECURITY: Set restrictive file permissions
            chmod($publicPath, 0644);

            // Generate public URL
            $publicUrl = url('uploads/invoices/whatsapp/' . $filename);

            // Schedule cleanup of this file after 24 hours
            $this->scheduleFileCleanup($publicPath, $filename);

            Log::info('WhatsAppBot: PDF uploaded successfully', [
                'invoice_no' => $safeInvoiceNo,
                'filename' => $filename
            ]);

            return $publicUrl;

        } catch (\Exception $e) {
            Log::error('WhatsAppBot: Error uploading PDF', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Schedule file cleanup after specified time
     *
     * @param string $filePath Full path to file
     * @param string $filename Sanitized filename for cache key
     * @param int $hoursDelay Hours to wait before cleanup
     */
    protected function scheduleFileCleanup($filePath, $filename, $hoursDelay = 24)
    {
        try {
            // SECURITY: Validate file path
            $realPath = realpath($filePath);
            $allowedBase = realpath(public_path('uploads/invoices/whatsapp'));
            
            if (!$realPath || !$allowedBase || strpos($realPath, $allowedBase) !== 0) {
                Log::warning('WhatsAppBot: Invalid file path for cleanup', [
                    'file' => $filePath
                ]);
                return;
            }

            // Use sanitized filename for cache key
            $cacheKey = 'whatsapp_cleanup_' . md5($filename);
            
            \Cache::put($cacheKey, [
                'file' => $realPath,
                'scheduled_at' => time(),
                'cleanup_at' => time() + ($hoursDelay * 3600)
            ], $hoursDelay * 60);

            Log::info('WhatsAppBot: File cleanup scheduled', [
                'filename' => $filename,
                'cleanup_in_hours' => $hoursDelay
            ]);

        } catch (\Exception $e) {
            Log::warning('WhatsAppBot: Could not schedule file cleanup', [
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Log invoice send attempt
     *
     * @param object $transaction
     * @param string $customerPhone
     * @param string $response
     * @param string $message
     * @param string $pdfFilename
     * @return WhatsAppInvoiceLog|null
     */
    protected function logInvoiceSend($transaction, $customerPhone, $response, $message = null, $pdfFilename = null)
    {
        try {
            // Get customer name from transaction
            $customerName = null;
            if ($transaction->contact) {
                $customerName = $transaction->contact->name;
            }

            // Create log entry
            $log = WhatsAppInvoiceLog::create([
                'business_id' => $transaction->business_id,
                'transaction_id' => $transaction->id,
                'invoice_no' => $transaction->invoice_no,
                'customer_phone' => $customerPhone,
                'customer_name' => $customerName,
                'message' => $message,
                'pdf_filename' => $pdfFilename,
                'status' => 'pending',
                'api_response' => null,
                'error_message' => null,
                'sent_at' => null,
                'retry_count' => 0,
            ]);

            // Update status based on response
            if ($response) {
                $log->markAsSent($response);
            } else {
                $log->markAsFailed('No API response received');
            }

            return $log;

        } catch (\Exception $e) {
            Log::warning('WhatsAppBot: Could not log invoice send', [
                'transaction_id' => $transaction->id,
                'customer_phone' => $customerPhone,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Generate invoice PDF using the existing invoice system
     *
     * @param object $transaction
     * @return string|null Path to generated PDF
     */
    public function generateInvoicePDF($transaction)
    {
        try {
            // Get required utilities
            $transactionUtil = new \App\Utils\TransactionUtil();
            
            // Set the current business context
            $business_id = $transaction->business_id;
            
            // Create a temporary session context for PDF generation
            $originalSession = session()->get('user.business_id');
            session()->put('user.business_id', $business_id);

            // Get PDF contents using the existing system
            $receipt_contents = $transactionUtil->getPdfContentsForGivenTransaction($business_id, $transaction->id);
            $receipt_details = $receipt_contents['receipt_details'];
            $location_details = $receipt_contents['location_details'];
            $is_email_attachment = false;

            // Generate PDF HTML content
            $body = view('sale_pos.receipts.download_pdf')
                ->with(compact('receipt_details', 'location_details', 'is_email_attachment'))
                ->render();

            // Create mPDF instance
            $mpdf = new \Mpdf\Mpdf([
                'tempDir' => public_path('uploads/temp'),
                'mode' => 'utf-8',
                'autoScriptToLang' => true,
                'autoLangToFont' => true,
                'autoVietnamese' => true,
                'autoArabic' => true,
                'margin_top' => 8,
                'margin_bottom' => 8,
                'format' => 'A4',
            ]);

            $mpdf->useSubstitutions = true;
            $mpdf->SetWatermarkText($receipt_details->business_name, 0.1);
            $mpdf->showWatermarkText = true;
            $mpdf->SetTitle('INVOICE-' . $receipt_details->invoice_no . '.pdf');
            $mpdf->WriteHTML($body);

            // Save PDF to temporary file
            $tempDir = storage_path('app/temp/whatsapp_invoices');
            if (!is_dir($tempDir)) {
                mkdir($tempDir, 0755, true);
            }

            $filename = 'invoice_' . $transaction->invoice_no . '_' . time() . '.pdf';
            $tempPath = $tempDir . '/' . $filename;
            
            $mpdf->Output($tempPath, 'F');

            // Restore original session
            if ($originalSession) {
                session()->put('user.business_id', $originalSession);
            }

            Log::info('WhatsAppBot: Invoice PDF generated successfully', [
                'transaction_id' => $transaction->id,
                'invoice_no' => $transaction->invoice_no,
                'pdf_path' => $tempPath
            ]);

            return $tempPath;

        } catch (\Exception $e) {
            Log::error('WhatsAppBot: Error generating invoice PDF', [
                'transaction_id' => $transaction->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }

    /**
     * Clean up old invoice PDFs from public storage
     */
    public function cleanupOldInvoices()
    {
        try {
            $publicDir = public_path('uploads/invoices/whatsapp');
            
            if (!is_dir($publicDir)) {
                return;
            }

            $files = glob($publicDir . '/invoice_*.pdf');
            $cleanedCount = 0;

            foreach ($files as $file) {
                // Remove files older than 24 hours
                if (file_exists($file) && (time() - filemtime($file)) > (24 * 3600)) {
                    unlink($file);
                    $cleanedCount++;
                }
            }

            if ($cleanedCount > 0) {
                Log::info('WhatsAppBot: Cleaned up old invoice PDFs', [
                    'files_removed' => $cleanedCount
                ]);
            }

        } catch (\Exception $e) {
            Log::error('WhatsAppBot: Error cleaning up old invoices', [
                'error' => $e->getMessage()
            ]);
        }
    }
}
