<?php

namespace App\Http\Controllers;
use Illuminate\Support\Str;
use App\Helpers\Helpers;
// Removed unused Module import
use App\Models\Package;
use App\Models\Transaction;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Http;

class TransactionController extends Controller
{
  public function list()
  {
    if (!Helpers::isEx()) {
      return redirect()->back();
    }
    if (empty(Helpers::setting('currencyCode'))) {
      $currencyCode = 'USD';
    } else {
      $currencyCode = Helpers::setting('currencyCode');
    }
    $user = auth()->user();
    $walletBalance = $user->transactions()->sum('amount');
    $transactions = $user
      ->transactions()
      ->orderBy('id', 'desc')
      ->paginate(50);
    return view('/user-transaction', compact('walletBalance', 'transactions', 'currencyCode'));
  }

  public static function store($code, $from_id, $to_id, $type, $amount, $data, $memo = null)
  {
    // $from_id transaction (minus)
    $transaction = new Transaction();
    $transaction->code = $code;
    $transaction->user_id = $from_id;
    $transaction->user2_id = $to_id;
    $transaction->type = $type;
    $transaction->amount = $amount * -1;
    $transaction->memo = $memo;
    $transaction->data = $data;
    $transaction->save();

    // $to_id transaction (plus)
    $transaction = new Transaction();
    $transaction->code = $code;
    $transaction->user_id = $to_id;
    $transaction->user2_id = $from_id;
    $transaction->type = $type;
    $transaction->amount = $amount;
    $transaction->memo = $memo;
    $transaction->data = $data;
    $transaction->save();

    File::ensureDirectoryExists(public_path("users/{$from_id}/transactions"), 0755, true);
    file_put_contents(public_path("users/{$from_id}/transactions/{$type}_{$code}.json"), $data);
    File::ensureDirectoryExists(public_path("users/{$to_id}/transactions"), 0755, true);
    file_put_contents(public_path("users/{$to_id}/transactions/{$type}_{$code}.json"), $data);
  }

  public static function autoRenew()
  {
    echo 'OK';
    self::renew();
    echo 'SUCCESS';
  }

  // public function autoRenew(Request $request)
  // {
  //   echo 'OK';
  //   if ($request->input('awg_orderId') !== Helpers::setting('awg_orderId')) {
  //     return;
  //   }
  //   self::renew();
  //   echo 'SUCCESS';
  // }

  public static function renew()
  {
    if (!Helpers::isEx()) {
      return;
    }
    $users = User::all();
    foreach ($users as $user) {
      self::renewPerUser($user->id);
    }
  }

  public static function renewPerUser($user_id)
  {
    if (!Helpers::isEx()) {
      return;
    }
    $user = User::where('id', '=', $user_id)->first();
    $expired = $user->role !== 'admin' && $user->billing_end < Carbon::now() ? true : false;
    if ($expired === true && $user->package->rate_monthly > 0) {
      $daysTotal =
        Carbon::now()
          ->startOfMonth()
          ->diffInseconds(Carbon::now()->endOfMonth(), false) / 86400;
      $daysRemaining = Carbon::now()->diffInseconds(Carbon::now()->endOfMonth(), false) / 86400;
      $prorated = ($daysRemaining / $daysTotal) * $user->package->rate_monthly;
      $walletBalance = $user->transactions()->sum('amount');
      if ($prorated > $walletBalance) {
        return;
      }
      $admin_id = 1;
      $transactionCode = Carbon::now()->timestamp . rand(100, 999);
      $transactionData = [
        'transactionCode' => $transactionCode,
        'description' => 'renew ' . $user->package->name . ' @ ' . $prorated,
      ];
      self::store($transactionCode, $user->id, $admin_id, 'renew', $prorated, json_encode($transactionData));

      // user table update
      $user->billing_start = Carbon::now()->startOfMonth();
      $user->billing_end = Carbon::now()->endOfMonth();
      $user->save();
    }
  }

  public function topup(Request $request)
  {
    if (!Helpers::isEx()) {
      return;
    }
    
    $gateway = $request->input('gateway');
    $userId = $request->input('user_id');
    $amount = $request->input('topupValue');
    
    // Validate inputs
    if (empty($gateway) || empty($userId) || empty($amount) || $amount <= 0) {
      return redirect()->back()->with(['danger_alert' => __('Invalid payment details.')]);
    }
    
    // Handle offline payment - check both OfflinePayment and offline
    if (strtolower($gateway) === 'offlinepayment' || strtolower($gateway) === 'offline') {
      return $this->offlinePayment($request);
    }
    
    // Get payment gateway from database
    $paymentGateway = \App\Models\PaymentGateway::where('name', strtolower($gateway))
      ->where('is_active', true)
      ->first();
    
    if ($paymentGateway) {
      // Route to specific gateway handler
      return $this->processGatewayPayment($request, $paymentGateway);
    }
    
    // Fallback to module-based gateways
    $modules = Module::getByStatus(1);
    foreach ($modules as $module) {
      if ($module->getName() === $gateway) {
        return redirect()->route($module->getName(), [
          'gateway' => $gateway,
          'user_id' => $userId,
          'user_name' => $request->input('user_name'),
          'currencyCode' => $request->input('currencyCode'),
          'topupValue' => $amount,
          'userGopayId' => $request->input('userGopayId'),
        ]);
      }
    }
    
    // Gateway not found or not active
    return redirect()->back()->with(['danger_alert' => __('Payment gateway not available.')]);
  }
  
  /**
   * Process payment through gateway
   */
  protected function processGatewayPayment($request, $paymentGateway)
  {
    $userId = $request->input('user_id');
    $amount = $request->input('topupValue');
    $currencyCode = $request->input('currencyCode', 'USD');
    
    // Create pending transaction
    $transaction = new \App\Models\Transaction();
    $transaction->user_id = $userId;
    $transaction->type = 'topup';
    $transaction->amount = 0; // Will be credited after payment confirmation
    $transaction->payment_gateway = $paymentGateway->name;
    $transaction->gateway_reference = 'PENDING_' . strtoupper(Str::random(16));
    $transaction->gateway_status = 'pending';
    $transaction->data = json_encode([
      'amount' => $amount,
      'currency' => $currencyCode,
      'user_name' => $request->input('user_name'),
    ]);
    $transaction->save();
    
    // Route to specific gateway
    switch ($paymentGateway->name) {
      case 'paystack':
        return $this->initializePaystack($request, $paymentGateway, $transaction);
      
      case 'flutterwave':
        return $this->initializeFlutterwave($request, $paymentGateway, $transaction);
      
      case 'paypal':
        return $this->initializePayPal($request, $paymentGateway, $transaction);
      
      case 'stripe':
        return $this->initializeStripe($request, $paymentGateway, $transaction);
      
      default:
        return redirect()->back()->with(['danger_alert' => __('Payment gateway not implemented yet.')]);
    }
  }
  
  /**
   * Initialize Paystack payment
   */
  protected function initializePaystack($request, $gateway, $transaction)
  {
    $settings = $gateway->settings ?? [];
    $secretKey = $gateway->secret_key;
    
    if (empty($secretKey)) {
      return redirect()->back()->with(['danger_alert' => __('Payment gateway not configured.')]);
    }
    
    $amount = $request->input('topupValue') * 100; // Convert to kobo/cents
    $email = \App\Models\User::find($request->input('user_id'))->email;
    
    $url = "https://api.paystack.co/transaction/initialize";
    $fields = [
      'email' => $email,
      'amount' => $amount,
      'currency' => $request->input('currencyCode', 'NGN'),
      'reference' => $transaction->gateway_reference,
      'callback_url' => route('transaction.topupSuccess'),
      'metadata' => [
        'user_id' => $request->input('user_id'),
        'transaction_id' => $transaction->id,
        'type' => 'topup'
      ]
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($fields));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
      "Authorization: Bearer $secretKey",
      "Content-Type: application/json",
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $result = curl_exec($ch);
    curl_close($ch);
    
    $response = json_decode($result);
    
    if ($response && $response->status) {
      return redirect($response->data->authorization_url);
    }
    
    return redirect()->back()->with(['danger_alert' => __('Failed to initialize payment.')]);
  }
  
  /**
   * Initialize Flutterwave payment
   */
  protected function initializeFlutterwave($request, $gateway, $transaction)
  {
    // Placeholder for Flutterwave implementation
    return redirect()->back()->with(['info_alert' => __('Flutterwave integration coming soon.')]);
  }
  
  /**
   * Initialize PayPal payment
   */
  protected function initializePayPal($request, $gateway, $transaction)
  {
    // Placeholder for PayPal implementation
    return redirect()->back()->with(['info_alert' => __('PayPal integration coming soon.')]);
  }
  
  /**
   * Initialize Stripe payment
   */
  protected function initializeStripe($request, $gateway, $transaction)
  {
    // Placeholder for Stripe implementation
    return redirect()->back()->with(['info_alert' => __('Stripe integration coming soon.')]);
  }

  public function topupSuccess(Request $request)
  {
    if (!Helpers::isEx()) {
      return;
    }
    // $verified = $this->verifyPayment($request->gateway, $request->pg_tr_id);
    // if (empty($verified)) return response()->json(['success' => false, 'message' => 'Payment cannot be verified!']);
    $bank_id = 2;
    self::store($request->pg_tr_id, $bank_id, $request->user_id, 'topup', $request->topupValue, $request->pg_response);
    self::renewPerUser($request->user_id);
  }

  public function mpSuccess(Request $request)
  {
    if (!Helpers::isEx()) {
      return;
    }
    // $verified = $this->verifyPayment('Mercadopago', $request->payment_id);
    // if (empty($verified)) return response()->json(['success' => false, 'message' => 'Payment cannot be verified!']);
    $pg_response = [
      'payment_gateway' => 'Mercadopago',
      'mercadopago_payment_id' => $request->payment_id,
      'mercadopago_payment_status' => $request->status,
      'mercadopago_external_reference' => $request->external_reference,
      'mercadopago_merchant_order_id' => $request->merchant_order_id,
    ];
    $ext_ref = json_decode($request->external_reference, true);
    $bank_id = 2;
    self::store($request->payment_id, $bank_id, $ext_ref['user_id'], 'topup', $ext_ref['topup_value'], json_encode($pg_response));
    self::renewPerUser($ext_ref['user_id']);
    return redirect()
      ->route('transaction.list', $ext_ref['user_id'])
      ->with(['success_alert' => __('Top up successful!')]);
  }

  public function stripeSuccess(Request $request)
  {
    if (!Helpers::isEx()) {
      return;
    }
    // $verified = $this->verifyPayment('stripe', $request->pg_tr_id);
    // if (empty($verified)) return response()->json(['success' => false, 'message' => 'Payment cannot be verified!']);
    $pg_response = [
      'payment_gateway' => 'Stripe',
      'stripe_session_id' => $request->pg_tr_id,
      'stripe_payment_amount' => $request->topupValue,
    ];
    $bank_id = 2;
    self::store($request->pg_tr_id, $bank_id, $request->user_id, 'topup', $request->topupValue, json_encode($pg_response));
    self::renewPerUser($request->user_id);
    return redirect()
      ->route('transaction.list', $request->user_id)
      ->with(['success_alert' => __('Top up successful!')]);
  }

  public static function gopaySuccess($data)
  {
    if (!Helpers::isEx()) {
      return;
    }
    $bank_id = 2;
    self::store($data['pg_tr_id'], $bank_id, $data['user_id'], 'topup', $data['topupValue'], $data['pg_response']);
    self::renewPerUser($data['user_id']);
  }

  public function upgrade(Request $request)
  {
    if (!Helpers::isEx()) {
      return redirect()->back();
    }
    $admin_id = 1;
    $user = auth()->user();
    $daysTotal =
      Carbon::now()
        ->startOfMonth()
        ->diffInseconds(Carbon::now()->endOfMonth(), false) / 86400;
    $daysRemaining = Carbon::now()->diffInseconds(Carbon::now()->endOfMonth(), false) / 86400;
    $billing_interval = $request->input('billing_interval');
    $package = Package::where('id', '=', $request->input('package_id'))->first();
    if ($user->package->name === 'trial' || $user->package->id === 2) {
      $prorated = $user->package->rate_monthly > $package->rate_monthly ? 0 : ($daysRemaining / $daysTotal) * $package->rate_monthly;
    } else {
      $prorated = $user->package->rate_monthly > $package->rate_monthly ? 0 : $package->rate_monthly - $user->package->rate_monthly;
    }
    if ($prorated > $request->input('walletBalance')) {
      return redirect()
        ->route('user.show', ['user_id' => $user->id])
        ->with(['danger_alert' => __('Your wallet balance is not enough. Please top up first!')]);
    }

    $transactionCode = Carbon::now()->timestamp . rand(100, 999);
    $transactionData = [
      'transactionCode' => $transactionCode,
      'description' => 'from ' . $user->package->name . ' to ' . $package->name,
    ];
    self::store($transactionCode, $user->id, $admin_id, 'upgrade', $prorated, json_encode($transactionData));

    // user table update
    $user->package_id = $package->id;
    $user->billing_interval = $billing_interval;
    $user->billing_start = Carbon::now()->startOfMonth();
    $user->billing_end = Carbon::now()->endOfMonth();
    $user->save();

    return redirect()
      ->route('transaction.list', $user->id)
      ->with(['success_alert' => __('Upgrade successful!')]);
  }

  public function transfer(Request $request)
  {
    if (!Helpers::isEx()) {
      return redirect()->back();
    }
    $from = User::where('name', '=', $request->input('from_name'))->first();
    $to = User::where('name', '=', $request->input('to_name'))->first();
    if (empty($to->id)) {
      return redirect()->back()->with(['danger_alert' => __('Cannot find Username') . ': ' . $request->input('to_name')]);
    }
    // Validate OTP if WA verification is enabled and adminSender is provided
    if (filter_var(Helpers::setting('waVerification'), FILTER_VALIDATE_BOOLEAN) && $from->role !== 'admin' && !empty($request->input('adminSender'))) {
      if ($from->otp !== $request->input('otp')) {
        return redirect()
          ->back()
          ->with(['danger_alert' => __('OTP code is invalid.')]);
      }
    }
    $from->otp = null;
    $from->save();
    $fromBalance = $from->transactions()->sum('amount');
    if ($from->role !== 'admin' && $request->input('amount') > $fromBalance) {
      return redirect()
        ->back()
        ->with(['danger_alert' => __('Balance is not enough.')]);
    }
    $transactionCode = Carbon::now()->timestamp . rand(100, 999);
    $transactionData = [
      'transactionCode' => $transactionCode,
      'description' => 'from ' . $request->input('from_name') . ' to ' . $request->input('to_name'),
    ];
    self::store($transactionCode, $from->id, $to->id, 'transfer', $request->input('amount'), json_encode($transactionData), $request->input('memo'));
    self::renewPerUser($to->id);
    return redirect()
      ->route('transaction.list', $from->id)
      ->with(['success_alert' => __('Transfer successful!')]);
  }

  public function paymentResponse(Request $request)
  {
    $filename = public_path("users/1/payment/" . date('Y-m-d-H-i-s') . ".json");
    $data = json_encode($request->all(), JSON_PRETTY_PRINT);
    file_put_contents($filename, $data);
    return response('Data received successfully.', 200)->header('Content-Type', 'text/plain');
  }

  private function verifyPayment($gateway, $transactionId) {
    if (strtolower($gateway) === 'mercadopago') {
      $mpAccessToken = Helpers::setting('mpAccessToken');
      require_once Module::getModulePath('Mercadopago') . 'vendor/autoload.php';
      \MercadoPago\MercadoPagoConfig::setAccessToken($mpAccessToken);
      \MercadoPago\MercadoPagoConfig::setRuntimeEnviroment(\MercadoPago\MercadoPagoConfig::LOCAL);
      $client = new \MercadoPago\Client\Payment\PaymentClient();
      $response = $client->get($transactionId);
      // file_put_contents(public_path('response.json'), json_encode($response));
      if (empty($response->status)) return false;
      if (strtolower($response->status) === 'approved') return true;
    } else if (strtolower($gateway) === 'midtrans') {
      $midtransServerKey = Helpers::setting('midtransServerKey');
      require_once Module::getModulePath('Midtrans') . 'vendor/autoload.php';
      \Midtrans\Config::$serverKey = $midtransServerKey;
      \Midtrans\Config::$isProduction = str_starts_with($midtransServerKey, 'SB-') ? false : true;
  $response = \Midtrans\Transaction::status($transactionId);
      // file_put_contents(public_path('response.json'), json_encode($response));
      if (str_starts_with(strtolower($response?->status_message), 'success')) return true;
      else return false;
    } else if (strtolower($gateway) === 'paypal') {
      $paypalSandbox = config('app.PAYPAL_SANDBOX') ?? false;
      $paypalClientId = Helpers::setting('paypalClientId');
      $paypalSecretKey = Helpers::setting('paypalSecretKey');
      $response = Http::withoutVerifying()
      ->withOptions(["verify"=>false])
      ->withBasicAuth($paypalClientId, $paypalSecretKey)
      ->asForm()
      ->post($paypalSandbox ? 'https://api-m.sandbox.paypal.com/v1/oauth2/token' : 'https://api-m.paypal.com/v1/oauth2/token', ['grant_type' => 'client_credentials']);
      $accessToken = json_decode($response)?->access_token;
      if (empty($accessToken)) return false;
      $response = Http::withoutVerifying()
      ->withOptions(["verify"=>false])
      ->withToken($accessToken)
      ->get($paypalSandbox ? "https://api-m.sandbox.paypal.com/v1/payments/sale/{$transactionId}" : "https://api-m.paypal.com/v1/payments/sale/{$transactionId}");
      // file_put_contents(public_path('response.json'), $response);
      $res = json_decode($response, true);
      if (empty($res['state'])) return false;
      if (strtolower($res['state']) === 'completed') return true;
    } else if (strtolower($gateway) === 'pix') {
      return true;
    } else if (strtolower($gateway) === 'razorpay') {
      $razorpayKeyId = Helpers::setting('razorpayKeyId');
      $razorpayKeySecret = Helpers::setting('razorpayKeySecret');
      require_once Module::getModulePath('Razorpay') . 'vendor/autoload.php';
  $api = new \Razorpay\Api\Api($razorpayKeyId, $razorpayKeySecret);
      $response = $api->payment->fetch($transactionId);
      // file_put_contents(public_path('response.json'), json_encode($response));
      if (empty($response->status)) return false;
      if (strtolower($response->status) == 'captured') return true;
    } else if (strtolower($gateway) === 'stripe') {
      $stripeSecret = Helpers::setting('stripeSecret');
      require_once Module::getModulePath('Stripe') . 'vendor/autoload.php';
  $stripe = new \Stripe\StripeClient($stripeSecret);
      $response = $stripe->checkout->sessions->retrieve($transactionId);
      // file_put_contents(public_path('response.json'), json_encode($response));
      if (empty($response->payment_status)) return false;
      if (strtolower($response->status) == 'paid') return true;
    }
    return false;
  }

  public function offlinePayment(Request $request)
  {
    // Get offline payment gateway settings
    $offlineGateway = \App\Models\PaymentGateway::where('name', 'offline')->first();
    
    if (!$offlineGateway) {
      return redirect()->back()->with(['danger_alert' => __('Offline payment gateway not configured.')]);
    }
    
    if (!$offlineGateway->is_active) {
      return redirect()->back()->with(['danger_alert' => __('Offline payment gateway is currently disabled.')]);
    }
    
    $offlineSettings = $offlineGateway->settings ?? [];
    
    // Get user from request (for admin) or auth user
    $userId = $request->input('user_id') ?? auth()->user()->id;
    $user = \App\Models\User::findOrFail($userId);
    
    // Get amount from topupValue or amount field
    $amount = $request->input('topupValue') ?? $request->input('amount');
    
    if (empty($amount) || $amount <= 0) {
      return redirect()->back()->with(['danger_alert' => __('Invalid amount specified.')]);
    }
    
    // Validate amount is numeric
    if (!is_numeric($amount)) {
      return redirect()->back()->with(['danger_alert' => __('Amount must be a valid number.')]);
    }
    
    // Generate reference
    $reference = $request->input('reference');
    if (empty($reference)) {
      $reference = 'OFF_' . strtoupper(Str::random(16));
    } else {
      // Sanitize user input reference
      $reference = 'OFF_' . strtoupper(Str::random(8)) . '_' . substr(preg_replace('/[^A-Za-z0-9]/', '', $reference), 0, 20);
    }
    
    // Check if proof is required
    $requireProof = $offlineSettings['require_proof'] ?? true;
    
    // Handle file upload
    $proofPath = null;
    if ($request->hasFile('proof_of_payment')) {
      $file = $request->file('proof_of_payment');
      
      // Validate file
      $validatedFile = $request->validate([
        'proof_of_payment' => 'required|file|mimes:jpeg,jpg,png,gif,pdf|max:5120' // 5MB max
      ]);
      
      $filename = 'proof_' . $userId . '_' . time() . '_' . Str::random(8) . '.' . $file->getClientOriginalExtension();
      $proofPath = $file->storeAs('proofs', $filename, 'public');
    } elseif ($requireProof) {
      // Only require proof if this is actually an offline payment
      return redirect()->back()
        ->withInput()
        ->with(['danger_alert' => __('Proof of payment is required. Please upload a screenshot or PDF of your payment receipt.')]);
    }
    
    // Create transaction
    $transaction = new \App\Models\Transaction();
    $transaction->user_id = $user->id;
    $transaction->type = 'topup';
    $transaction->amount = 0; // Set to 0 until admin approves - will be updated on approval
    $transaction->payment_gateway = 'offline';
    $transaction->gateway_reference = $reference;
    $transaction->gateway_status = 'pending';
    $transaction->confirmed = false; // Explicitly set to false until admin approval
    $transaction->proof_of_payment = $proofPath;
    $transaction->data = json_encode([
      'submitted_at' => now()->toDateTimeString(),
      'requested_amount' => $amount, // Store the requested amount here
      'ip_address' => $request->ip(),
      'user_agent' => $request->userAgent(),
      'notes' => $request->input('reference')
    ]);
    $transaction->save();
    
    // Send notification email to admin (optional)
    try {
      // You can implement email notification here
      // Mail::to(config('mail.admin_email'))->send(new OfflinePaymentReceived($transaction));
    } catch (\Exception $e) {
      \Log::error('Failed to send offline payment notification: ' . $e->getMessage());
    }
    
    // Return to user profile with success message
    return redirect()
      ->route('user.show', ['user_id' => $user->id])
      ->with(['success_alert' => __('Offline payment submitted successfully! Your payment is being reviewed. Your wallet will be credited after admin approval.')]);
  }

  /**
   * Admin: List all offline payments
   */
  public function listOfflinePayments()
  {
    if (auth()->user()->role !== 'admin') {
      return redirect()->back()->with(['danger_alert' => __('Unauthorized access.')]);
    }

    $payments = Transaction::with('user')
      ->where('payment_gateway', 'offline')
      ->orderBy('id', 'desc')
      ->get();

    $pendingCount = Transaction::where('payment_gateway', 'offline')
      ->where('gateway_status', 'pending')
      ->count();

    $approvedCount = Transaction::where('payment_gateway', 'offline')
      ->whereIn('gateway_status', ['success', 'approved'])
      ->count();

    $rejectedCount = Transaction::where('payment_gateway', 'offline')
      ->whereIn('gateway_status', ['rejected', 'failed'])
      ->count();

    $totalAmount = Transaction::where('payment_gateway', 'offline')
      ->where('gateway_status', 'pending')
      ->sum('amount');

    return view('offline-payments', compact(
      'payments',
      'pendingCount',
      'approvedCount',
      'rejectedCount',
      'totalAmount'
    ));
  }

  /**
   * Admin: Approve offline payment
   */
  public function approveOfflinePayment($transaction_id)
  {
    if (auth()->user()->role !== 'admin') {
      return redirect()->back()->with(['danger_alert' => __('Unauthorized access.')]);
    }

    $transaction = Transaction::findOrFail($transaction_id);

    if ($transaction->payment_gateway !== 'offline') {
      return redirect()->back()->with(['danger_alert' => __('Invalid transaction type.')]);
    }

    if ($transaction->gateway_status !== 'pending') {
      return redirect()->back()->with(['danger_alert' => __('Transaction already processed.')]);
    }

    // Get user
    $user = \App\Models\User::findOrFail($transaction->user_id);
    
    // Get the requested amount from transaction data
    $transactionData = json_decode($transaction->data, true);
    $requestedAmount = $transactionData['requested_amount'] ?? 0;
    
    if ($requestedAmount <= 0) {
      return redirect()->back()->with(['danger_alert' => __('Invalid transaction amount.')]);
    }
    
    // Update transaction with approved amount
    $transaction->amount = $requestedAmount;
    $transaction->gateway_status = 'approved';
    $transaction->confirmed = true;
    $transaction->data = json_encode(array_merge($transactionData, [
      'approved_by' => auth()->user()->name,
      'approved_at' => now()->toDateTimeString(),
      'admin_id' => auth()->user()->id
    ]));
    $transaction->save();

    // Note: Wallet balance is calculated from confirmed transactions
    // No need to update user.amount as it doesn't exist
    
    // Log the approval
    \Log::info("Offline payment approved: Transaction #{$transaction->id}, User #{$user->id}, Amount: {$requestedAmount}");

    // Send notification to user (optional)
    try {
      // You can implement email/SMS notification here
      // Mail::to($user->email)->send(new PaymentApproved($transaction));
    } catch (\Exception $e) {
      \Log::error('Failed to send approval notification: ' . $e->getMessage());
    }

    return redirect()->back()->with([
      'success_alert' => __('Payment approved successfully! Amount of :amount has been added to :user\'s wallet.', [
        'amount' => number_format($requestedAmount, 2),
        'user' => $user->name
      ])
    ]);
  }

  /**
   * Admin: Reject offline payment
   */
  public function rejectOfflinePayment(Request $request, $transaction_id)
  {
    if (auth()->user()->role !== 'admin') {
      return redirect()->back()->with(['danger_alert' => __('Unauthorized access.')]);
    }

    $transaction = Transaction::findOrFail($transaction_id);

    if ($transaction->payment_gateway !== 'offline') {
      return redirect()->back()->with(['danger_alert' => __('Invalid transaction type.')]);
    }

    if ($transaction->gateway_status !== 'pending') {
      return redirect()->back()->with(['danger_alert' => __('Transaction already processed.')]);
    }

    $reason = $request->input('reason', 'No reason provided');
    
    // Get user for notification
    $user = \App\Models\User::findOrFail($transaction->user_id);

    // Update transaction status
    $transaction->gateway_status = 'rejected';
    $transaction->confirmed = false;
    $transaction->data = json_encode([
      'rejected_by' => auth()->user()->name,
      'rejected_at' => now()->toDateTimeString(),
      'admin_id' => auth()->user()->id,
      'reason' => $reason
    ]);
    $transaction->save();
    
    // Log the rejection
    \Log::info("Offline payment rejected: Transaction #{$transaction->id}, User #{$user->id}, Amount: {$transaction->amount}, Reason: {$reason}");

    // Send notification to user (optional)
    try {
      // You can implement email/SMS notification here
      // Mail::to($user->email)->send(new PaymentRejected($transaction, $reason));
    } catch (\Exception $e) {
      \Log::error('Failed to send rejection notification: ' . $e->getMessage());
    }

    return redirect()->back()->with([
      'success_alert' => __('Payment rejected successfully.')
    ]);
  }
}
