<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class TicketMessage extends Model
{
    use HasFactory;

    protected $fillable = [
        'ticket_id',
        'sender_type',
        'sender_id',
        'message',
        'attachments',
        'message_type',
        'is_read',
        'is_internal_note',
    ];

    protected $casts = [
        'attachments' => 'array',
        'is_read' => 'boolean',
        'is_internal_note' => 'boolean',
    ];

    // Relationships
    public function ticket()
    {
        return $this->belongsTo(SupportTicket::class);
    }

    public function sender()
    {
        // If sender_type is 'agent' or 'system', return User
        if (in_array($this->sender_type, ['agent', 'system']) && $this->sender_id) {
            return $this->belongsTo(User::class, 'sender_id');
        }

        return null;
    }

    // Scopes
    public function scopeUnread($query)
    {
        return $query->where('is_read', false);
    }

    public function scopeFromCustomer($query)
    {
        return $query->where('sender_type', 'customer');
    }

    public function scopeFromAgent($query)
    {
        return $query->where('sender_type', 'agent');
    }

    public function scopeFromSystem($query)
    {
        return $query->where('sender_type', 'system');
    }

    public function scopeInternalNotes($query)
    {
        return $query->where('is_internal_note', true);
    }

    public function scopePublicMessages($query)
    {
        return $query->where('is_internal_note', false);
    }

    public function scopeByType($query, $type)
    {
        return $query->where('message_type', $type);
    }

    public function scopeForTicket($query, $ticketId)
    {
        return $query->where('ticket_id', $ticketId);
    }

    // Methods
    public function markAsRead()
    {
        $this->is_read = true;
        $this->save();

        return $this;
    }

    public function isFromCustomer()
    {
        return $this->sender_type === 'customer';
    }

    public function isFromAgent()
    {
        return $this->sender_type === 'agent';
    }

    public function isFromSystem()
    {
        return $this->sender_type === 'system';
    }

    public function isInternalNote()
    {
        return $this->is_internal_note === true;
    }

    public function hasAttachments()
    {
        return !empty($this->attachments);
    }

    public function getFormattedMessageAttribute()
    {
        if ($this->isFromSystem()) {
            return '<em class="text-muted">' . e($this->message) . '</em>';
        }

        return nl2br(e($this->message));
    }

    public function getSenderNameAttribute()
    {
        if ($this->isFromCustomer()) {
            return $this->ticket->customer_name ?? $this->ticket->customer_phone;
        }

        if ($this->sender_id && in_array($this->sender_type, ['agent', 'system'])) {
            $user = User::find($this->sender_id);
            return $user ? $user->name : 'System';
        }

        return 'System';
    }
}
