<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('todos', function (Blueprint $table) {
            // Recurring Todos
            $table->boolean('is_recurring')->default(false)->after('is_completed');
            $table->enum('recurrence_type', ['daily', 'weekly', 'monthly', 'yearly'])->nullable()->after('is_recurring');
            $table->integer('recurrence_interval')->default(1)->after('recurrence_type'); // Every X days/weeks/months
            $table->json('recurrence_days')->nullable()->after('recurrence_interval'); // For weekly: [1,3,5] = Mon,Wed,Fri
            $table->dateTime('recurrence_end_date')->nullable()->after('recurrence_days');
            $table->unsignedBigInteger('parent_todo_id')->nullable()->after('recurrence_end_date'); // For recurring instances
            
            // File Attachments
            $table->json('attachments')->nullable()->after('parent_todo_id'); // Array of file paths
            
            // Sharing
            $table->json('shared_with')->nullable()->after('attachments'); // Array of user IDs
            $table->enum('share_permission', ['view', 'edit'])->default('view')->after('shared_with');
            
            // Categories and Tags
            $table->string('category')->nullable()->after('share_permission'); // Project, Personal, Work, etc.
            $table->json('tags')->nullable()->after('category'); // Array of tags
            
            // Multiple Reminders
            $table->json('reminder_times')->nullable()->after('tags'); // [60, 120, 1440] for multiple reminders
            $table->json('reminders_sent')->nullable()->after('reminder_times'); // Track which reminders were sent
            
            // Snooze Functionality
            $table->dateTime('snoozed_until')->nullable()->after('reminders_sent');
            $table->integer('snooze_count')->default(0)->after('snoozed_until');
            
            // Additional metadata
            $table->string('location')->nullable()->after('snooze_count');
            $table->json('metadata')->nullable()->after('location'); // For extensibility
            
            // Indexes for performance
            $table->index('is_recurring');
            $table->index('category');
            $table->index('parent_todo_id');
            $table->index('snoozed_until');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('todos', function (Blueprint $table) {
            $table->dropColumn([
                'is_recurring',
                'recurrence_type',
                'recurrence_interval',
                'recurrence_days',
                'recurrence_end_date',
                'parent_todo_id',
                'attachments',
                'shared_with',
                'share_permission',
                'category',
                'tags',
                'reminder_times',
                'reminders_sent',
                'snoozed_until',
                'snooze_count',
                'location',
                'metadata',
            ]);
        });
    }
};
