<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('support_tickets', function (Blueprint $table) {
            $table->id();
            $table->string('ticket_number')->unique();
            $table->unsignedBigInteger('device_id');
            $table->string('customer_phone');
            $table->string('customer_name')->nullable();
            $table->string('subject')->nullable();
            $table->text('description')->nullable();
            $table->enum('status', ['open', 'pending', 'in_progress', 'on_hold', 'resolved', 'closed'])->default('open');
            $table->enum('priority', ['low', 'normal', 'high', 'urgent'])->default('normal');
            $table->unsignedBigInteger('assigned_to')->nullable();
            $table->unsignedBigInteger('created_by')->nullable();
            $table->json('attachments')->nullable();
            $table->json('conversation_history')->nullable();
            $table->integer('lead_score')->default(0);
            $table->timestamp('first_response_at')->nullable();
            $table->timestamp('resolved_at')->nullable();
            $table->timestamp('last_activity_at')->nullable();
            $table->timestamps();

            $table->foreign('device_id')->references('id')->on('devices')->onDelete('cascade');
            $table->foreign('assigned_to')->references('id')->on('users')->onDelete('set null');
            $table->foreign('created_by')->references('id')->on('users')->onDelete('set null');
            
            $table->index(['device_id', 'customer_phone']);
            $table->index('status');
            $table->index('priority');
            $table->index('assigned_to');
            $table->index('created_at');
        });

        Schema::create('ticket_agents', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('user_id');
            $table->unsignedBigInteger('device_id');
            $table->enum('role', ['agent', 'supervisor', 'admin'])->default('agent');
            $table->enum('status', ['online', 'offline', 'busy', 'away'])->default('offline');
            $table->integer('max_tickets')->default(10);
            $table->integer('current_tickets')->default(0);
            $table->json('permissions')->nullable();
            $table->boolean('auto_assign')->default(true);
            $table->timestamps();

            $table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
            $table->foreign('device_id')->references('id')->on('devices')->onDelete('cascade');
            
            $table->unique(['user_id', 'device_id']);
            $table->index('status');
        });

        Schema::create('ticket_messages', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('ticket_id');
            $table->enum('sender_type', ['customer', 'agent', 'system']);
            $table->unsignedBigInteger('sender_id')->nullable();
            $table->text('message');
            $table->json('attachments')->nullable();
            $table->enum('message_type', ['text', 'image', 'document', 'video', 'audio', 'location'])->default('text');
            $table->boolean('is_read')->default(false);
            $table->boolean('is_internal_note')->default(false);
            $table->timestamps();

            $table->foreign('ticket_id')->references('id')->on('support_tickets')->onDelete('cascade');
            $table->foreign('sender_id')->references('id')->on('users')->onDelete('set null');
            
            $table->index('ticket_id');
            $table->index('is_read');
        });

        Schema::create('leads', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('device_id');
            $table->string('customer_phone');
            $table->string('customer_name')->nullable();
            $table->enum('source', ['whatsapp', 'ticket', 'manual'])->default('whatsapp');
            $table->enum('status', ['new', 'contacted', 'qualified', 'proposal', 'negotiation', 'converted', 'lost'])->default('new');
            $table->integer('lead_score')->default(0);
            $table->json('notes')->nullable();
            $table->json('custom_fields')->nullable();
            $table->unsignedBigInteger('assigned_to')->nullable();
            $table->unsignedBigInteger('ticket_id')->nullable();
            $table->timestamp('last_contact_at')->nullable();
            $table->timestamp('converted_at')->nullable();
            $table->timestamps();

            $table->foreign('device_id')->references('id')->on('devices')->onDelete('cascade');
            $table->foreign('assigned_to')->references('id')->on('users')->onDelete('set null');
            $table->foreign('ticket_id')->references('id')->on('support_tickets')->onDelete('set null');
            
            $table->index(['device_id', 'customer_phone']);
            $table->index('status');
            $table->index('lead_score');
            $table->index('assigned_to');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('leads');
        Schema::dropIfExists('ticket_messages');
        Schema::dropIfExists('ticket_agents');
        Schema::dropIfExists('support_tickets');
    }
};
